#ifndef PROFDIST_LIB_TRAITS_HPP_INCLUDED
#define PROFDIST_LIB_TRAITS_HPP_INCLUDED

/***************************************************************************
 *   Copyright (C) 2005 by Andreas Pokorny                                 *
 *   andreas.pokorny@biozentrum.uni-wuerzburg.de                           *
 *                                                                         *
 *   This file is part of profdist and cbcanalyzer                         *
 *                                                                         *
 *   Both profdist and cbcanalyzer are free software; you can redistribute *
 *   it and/or modify it under the terms of the GNU General Public License *
 *   as published by the Free Software Foundation; either version 2 of the *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   Profdist and cbcanalyzer are distributed in the hope that it will be  *
 *   useful, but WITHOUT ANY WARRANTY; without even the implied warranty   *
 *   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <boost/iterator/iterator_facade.hpp>
#include "fixed_matrix.h" 
#include "parsed_sequence.h" 
#include "dnamapper.h"
#include "rna_structure_mapper.hpp"
#include "protein_mapper.hpp"

namespace profdist {
 
/**
* concept of the trait class: a dna trait could look like that:
 */
#if 0
struct traits {
  static const size_t num_elements = 4;
  typedef char encoding_type;

  /**
   * The cursor is something like a read only iterator..
   */
  template<typename Iterator> struct cursor{};
  template<typename Iterator> 
  static encoding_type get_entry( sequence<Iterator> const& container, typename cursor<Iterator> const& container );
  template<typename Iterator> 
  static cursor<Iterator> begin( sequence<Iterator> const& seq );
  template<typename Iterator> 
  static cursor<Iterator> end( sequence<Iterator> const& seq );
};
#endif

struct rna_traits {
  typedef unsigned char encoding_type;
  static const size_t num_relevant_elements = 4; ///< Number of nucleotids for the acids (A,C,G and T).
  static const size_t num_elements = 6; ///< This is number of nucleotids (num_relevant_elements) plus the additional symbols N and -.
  static const size_t gap = 4;
  typedef fixed_matrix<double,num_relevant_elements,num_relevant_elements> rate_matrix;

  struct cursor
    : boost::iterator_facade<cursor,encoding_type, std::forward_iterator_tag,encoding_type>
  {
    typedef char* it_type; 
    typedef std::pair<it_type,it_type> range_type;
    typedef std::list<range_type>::const_iterator const_iterator;
    range_type range_it;
    const_iterator seq_it, end;
    cursor( sequence<it_type> const& seq, const_iterator it );
    encoding_type dereference()	const;
    bool equal( cursor const& other )const;
    void increment();
  };

  static cursor begin( sequence<char*> const& seq ) {
    return cursor( seq, seq.sequence_data.begin() );
  }
   
  static cursor end( sequence<char*> const& seq ){
    return cursor( seq, seq.sequence_data.end() );
  }

  static encoding_type get_entry( cursor const& c ) {
    return *c;
  }
  static char get_char( encoding_type  c ) {
    //return 'a'; /* Why always output an 'a'?  */
	return get_mapper_instance().code2dna[c];
  }
};

struct rna_structure_traits {
  typedef unsigned char encoding_type;

  static const size_t num_relevant_elements = 12; // ACGT * (.)
  static const size_t num_elements = 16; // ACGTN * (.) + gap ;
  static const size_t gap = 15;
  typedef fixed_matrix<double,num_relevant_elements,num_relevant_elements> rate_matrix;

  struct cursor
    : boost::iterator_facade<cursor,encoding_type, std::forward_iterator_tag,encoding_type> 
  {
    typedef char* it_type; 
    typedef sequence<it_type>::string_range range_type;
    typedef std::list<range_type>::const_iterator const_iterator;

    range_type seq_range_it, fold_range_it;
    const_iterator seq_it, fold_it, seq_end, fold_end;

    cursor( sequence<it_type>const& seq, const_iterator seq_it, const_iterator fold_it );
    encoding_type dereference()	const;
    bool equal( cursor const& other )const;
    void increment();
  };

  static cursor begin( sequence<char*> const& seq ) {
    return cursor( seq, seq.sequence_data.begin(), seq.char_fold_data.begin() );
  }
   
  static cursor end( sequence<char*> const& seq ){
    return cursor( seq, seq.sequence_data.end(), seq.char_fold_data.begin() );
  }

  static encoding_type get_entry( cursor const& c ) {
    return *c;
  }
  
  
  static char get_char( encoding_type  c ) {
    //return 'a';
	return rna_structure_mapper::get_instance().from_code(c).first;
  }
  
  static char get_fold_char( encoding_type c) {
  	return rna_structure_mapper::get_instance().from_code(c).second;
  }
  
  /*
  static std::pair<char, char> get_char(encoding_type c)
  {
  	return rna_structure_mapper::get_instance().from_code(c);
  }
  */

};

struct protein_traits {
  typedef unsigned char encoding_type;
  static const size_t num_relevant_elements = 20; // 20 amino acids without gap
  static const size_t num_elements = 22; // 20 amino acids, mixed acids and gap 
  static const size_t gap = 21;
  typedef fixed_matrix<double,num_relevant_elements,num_relevant_elements> rate_matrix;

  struct cursor
    : boost::iterator_facade<cursor,encoding_type, std::forward_iterator_tag,encoding_type> 
  {
    typedef char* it_type; 
    typedef sequence<it_type>::string_range range_type;
    typedef std::list<range_type>::const_iterator const_iterator;
    
    range_type range_it;
    const_iterator seq_it, end;

    cursor( sequence<it_type>const& seq, const_iterator it );
    encoding_type dereference()	const;
    bool equal( cursor const& other )const;
    void increment();
  };

  static cursor begin( sequence<char*> const& seq ) {
    return cursor( seq, seq.sequence_data.begin() );
  }
   
  static cursor end( sequence<char*> const& seq ){
    return cursor( seq, seq.sequence_data.end() );
  }

  static char get_char( encoding_type  c ) {
    //return 'a';
	return protein_mapper::get_instance().to_protein(c);
  }


  static encoding_type get_entry( cursor const& c ) {
    return *c;
  }
};

}

#endif

