#ifndef PROFDIST_PROFILE_INL_INCLUDED
#define PROFDIST_PROFILE_INL_INCLUDED

/***************************************************************************
 *   Copyright (C) 2005 by Andreas Pokorny                                 *
 *   andreas.pokorny@biozentrum.uni-wuerzburg.de                           *
 *                                                                         *
 *   This file is part of profdist and cbcanalyzer                         *
 *                                                                         *
 *   Both profdist and cbcanalyzer are free software; you can redistribute * 
 *   it and/or modify it under the terms of the GNU General Public License * 
 *   as published by the Free Software Foundation; either version 2 of the * 
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   Profdist and cbcanalyzer are distributed in the hope that it will be  *
 *   useful, but WITHOUT ANY WARRANTY; without even the implied warranty   *
 *   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
template<typename Traits>
profdist::Profile::Profile( AlignCode<Traits> const& source, tree_types::profile_map const& tree )
  : num_sites(source.get_num_sites()), num_profiles(tree.size()), num_rows( Traits::num_elements ), profile_array( num_profiles*num_sites*num_rows, 0.0 )
{
  inner_create_profiles( source, tree );
}

template<typename Traits>
profdist::Profile::Profile( profdist::alignment const& source, Traits const& trait, std::list<std::list<std::size_t> > const& profiles  )
{
  if( ! source.empty() ) {
    num_sites = source.alignment_length();
    num_profiles = profiles.size(); 
    num_rows = Traits::num_elements; 
    profile_array.resize( num_profiles*num_sites*num_rows, 0.0 );
    inner_create_profiles( source, trait, profiles );
  }
}

template<typename Traits>
profdist::Profile::Profile( AlignCode<Traits> const& source, std::list<std::list<std::size_t> > const& profiles )
  : num_sites(source.get_num_sites()), num_profiles(profiles.size()), num_rows( Traits::num_elements), profile_array( num_profiles*num_sites*num_rows, 0.0 )
{
  inner_create_profiles( source, profiles );
}


template<typename Traits>
void profdist::Profile::init( AlignCode<Traits> const& source, tree_types::profile_map const& tree )
{
  init( source.get_num_sites(), tree.size(), Traits::num_rows );
  inner_create_profiles( source, tree );
}
template<typename Traits>
void profdist::Profile::init( AlignCode<Traits> const& source, std::list< std::list<std::size_t> >  const& profiles )
{
  init( source.get_num_sites(), profiles.size(), Traits::num_rows );
  inner_create_profiles( source, profiles );
}

template<typename Traits>
void profdist::Profile::inner_create_profiles( profdist::alignment const& source, Traits const& traits,std::list< std::list<std::size_t> >  const& profiles )
{
  using namespace std;

  size_t prof_index = 0;
  size_t input_size = source.size();
  for( list<list<size_t> >::const_iterator it = profiles.begin(), e = profiles.end();  it != e; ++it, ++prof_index )
  {
    SingleProfile profile = get_profile( prof_index );

    size_t profile_size = it->size();

    // first get a sorted copy of our profile:
    size_t source_index = 0;
    std::list<size_t> prof_sorted = *it;
    prof_sorted.sort();
    alignment::const_iterator source_it = source.begin();

    //for each sequence in profile:
    for( std::list<size_t>::const_iterator prof_it = prof_sorted.begin(), prof_e = prof_sorted.end(); 
        prof_it != prof_e; ++prof_it ) 
    {

      // get the right sequence in our alignment file:
      std::advance( source_it, *prof_it - source_index );
      source_index = *prof_it;
      if( input_size <= source_index )
        throw std::runtime_error("Invalid sequence selected in profile");

      std::size_t base_index = 0;

      // for each string_range of that sequence:
      for( typename Traits::cursor it = Traits::begin( *source_it ), e = Traits::end( *source_it )
          ; it != e ; ++it, ++base_index )
        profile.get_value(base_index, *it ) += 1;
          
    }

    profile.normalize();
  }
}

template<typename Traits>
void profdist::Profile::inner_create_profiles( AlignCode<Traits> const& source, std::list< std::list<std::size_t> >  const& profiles )
{
  using namespace std;
  size_t prof_index = 0;
  for( list<list<size_t> >::const_iterator it = profiles.begin(), e = profiles.end();  it != e; ++it, ++prof_index )
  {
    SingleProfile profile = get_profile( prof_index );

//    std::vector<size_t> entries( num_sites, size_t(0) );
    size_t profile_size = it->size();
    // first gather information from the reference sequence 
    for( size_t i = 0; i != num_sites; ++i )  {
      size_t reference_element = source.get_reference_element( i );
      if( reference_element < num_rows )
        profile.get_value(i, reference_element ) = profile_size;
      else 
        profile.get_value(i, num_rows - 1 ) += profile_size;
    }

    handle_differences( profile, source, it->begin(), it->end());
    profile.normalize();
  }
}
template<typename Traits>
void profdist::Profile::process_difference( SingleProfile & profile, AlignCode<Traits> const& code, std::size_t index )
{
  for( typename AlignCode<Traits>::const_diff_iterator diff_it = code.begin_difference( index ), 
      diff_end = code.end_difference( index );
      diff_it != diff_end; ++diff_it ) {
    size_t ref_element = code.get_reference_element( diff_it->first );
    if( diff_it->second < num_rows ) 
      // .. add the difference to the reference sequence
      profile.get_value( diff_it->first, diff_it->second) += 1.0;
    else  
      profile.get_value( diff_it->first, num_rows - 1) += 1.0;

    if( ref_element < num_rows ) 
      // .. and remove the reference entries at that position: provided that the position had an entry in the ref. seq. at all!
      profile.get_value( diff_it->first, ref_element ) -= 1.0; 
    else 
      profile.get_value( diff_it->first, num_rows - 1) -= 1.0;
  }
}

template<typename Traits>
void profdist::Profile::inner_create_profiles( AlignCode<Traits> const& source, tree_types::profile_map const& tree )
{
  for( tree_types::profile_map::const_iterator it = tree.begin(), e = tree.end();  it != e; ++it )
  {
    SingleProfile profile = get_profile( it->first );
   // std::vector<size_t> entry_numbers( num_sites, size_t(0L) );
    
    if( it->second->is_leaf() ) {  // just a single sequence to process here 

      // first set as if we would process the reference sequence
      for( std::size_t i = 0; i != num_sites; ++i )  {
        size_t ref_element = source.get_reference_element( i );
        if( ref_element < num_rows ) 
          profile.get_value(i, ref_element ) = 1;
        else
          profile.get_value(i, num_rows - 1) = 1;
      }

      // if we in fact process a different then ..
      if( it->second->get_reference_position() != 0 )
      {
        // - 1 because difference vectors starts at zero but stores elements 1 to N-1 
        process_difference( profile, source, it->second->get_reference_position() - 1 /*, entry_numbers */);
      }
    }
    else {
      std::size_t set_size = it->second->get_split_set().size();

      // first gather information from the reference sequence 
      for( std::size_t i = 0; i != num_sites; ++i ) 
        if( source.get_reference_element( i ) < num_rows ) 
          profile.get_value(i, source.get_reference_element( i ) ) = set_size;
        else
          profile.get_value(i, num_rows - 1) = set_size;


      handle_differences( profile, source, it->second->get_split_set().begin(), it->second->get_split_set().end() /*, entry_numbers */ );

   //  profile.normalize( entry_numbers );
      profile.normalize();
    }
  }
}

#endif

