/***************************************************************************
 *   Copyright (C) 2006 by Andreas Pokorny                                 *
 *   andreas.pokorny@biozentrum.uni-wuerzburg.de                           *
 *                                                                         *
 *   This file is part of profdist and cbcanalyzer                         *
 *                                                                         *
 *   Both profdist and cbcanalyzer are free software; you can redistribute *
 *   it and/or modify it under the terms of the GNU General Public License *
 *   as published by the Free Software Foundation; either version 2 of the *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   Profdist and cbcanalyzer are distributed in the hope that it will be  *
 *   useful, but WITHOUT ANY WARRANTY; without even the implied warranty   *
 *   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef PROFDIST_PARSED_ALIGNMENT_HPP_INCLUDED
#define PROFDIST_PARSED_ALIGNMENT_HPP_INCLUDED

#include <boost/shared_ptr.hpp>
#include <list>
#include "parsed_sequence.h"

namespace profdist {

  struct resource{
    virtual char* begin() = 0;
    virtual char* end() = 0;
    virtual ~resource();
  };

  /**
   * The alignment structure contains an alignment file, or resource.. 
   */
  struct alignment {
    public:
      typedef boost::shared_ptr<resource> file_resource;
      typedef sequence<char*> value_type;
      typedef std::list<sequence<char*> > con_type;
      typedef con_type::iterator iterator;
      typedef con_type::const_iterator const_iterator;
    private:
      file_resource store;
      size_t sequence_length;
      std::list<value_type> sequences;
    public:
      alignment() 
        : sequence_length(0) {}
      alignment( file_resource const& st ) 
        : store(st), sequence_length(0) {}

      inline file_resource get_store() const { return store; }
      inline void clear() {
        sequence_length = 0;
        sequences.clear();
        store.reset();
      }

      inline void reset( file_resource const& st ) {
        sequence_length = 0;
        sequences.clear();
        store = st;
      }

      inline void pop_front() { if(size() == 1 ) store.reset(); sequences.pop_front(); }
      inline void pop_back() { if(size() == 1 ) store.reset(); sequences.pop_back(); }
      inline value_type const& front() const { return sequences.front(); }
      inline value_type & front() { return sequences.front(); }
      inline value_type const& back() const { return sequences.back(); }
      inline value_type & back() { return sequences.back(); }
      inline bool empty() const { return sequences.empty(); }
      inline size_t size() const { return sequences.size(); }
      inline size_t alignment_length() const { return sequence_length; }
      inline const_iterator begin() const {return sequences.begin();} 
      inline iterator begin() {return sequences.begin();} 
      inline const_iterator end() const {return sequences.end();} 
      inline iterator end() {return sequences.end();} 
      inline iterator push_back( value_type const& val ) { sequence_length = val.sequence_length; sequences.push_back(val);}
  };
}

#endif

