/**
 * @file matrix_eigenvalues_cache.h
 * @author Benjamin Ruderisch
 * @brief So far the taylor row as been used to compute the matrix
 * exponentials of a given matrix and a value t. This computation
 * can be accelerated by using the eigenvalues of this matrix.
 * If we want to use the eigenvalues for computing the matrix
 * exponential, we would have to refine the exisiting interfaces
 * for computing matrix exponentials. So we introduce here this
 * chache class, which stores the needed information (eigenvalues,
 * U, U^-1 and the matrix itself). We also define a global cache
 * which can be accessed by the getGlobalCache() function.
 * Calling this function will return a reference  to the global
 * MatrixEigenvaluesCache object. Using this reference the cache
 * information can be queried and accessed.
 */

#ifndef MATRIX_EIGENVALUES_CACHE_H_
#define MATRIX_EIGENVALUES_CACHE_H_

#include <list>
#include <boost/any.hpp>
#include "fixed_matrix.h"

namespace profdist {

using boost::any;
using boost::any_cast;
using boost::bad_any_cast;

/**
 * @brief The MatrixInfo class stores the eigenvalues plus U and U^-1
 * of a given matrix. This class is used to store the information
 * in the cache.
 */
template<typename ValueT, size_t Rows, size_t Columns>
struct MatrixInfo {
	typedef fixed_matrix<ValueT, Rows, Columns> matrix_type;
	
	MatrixInfo() : matrix(0.0F), U(0.0F), lambda(0.0F), U_inv(0.0F) {}
	
	MatrixInfo(const matrix_type& m) : matrix(m), U(0.0F), lambda(0.0F), U_inv(0.0F) {}
	
	MatrixInfo(const MatrixInfo<ValueT, Rows, Columns>& info) : matrix(info.matrix), U(info.U), lambda(info.lambda), U_inv(info.U_inv) {}
	
	const MatrixInfo& operator=(const MatrixInfo<ValueT, Rows, Columns>& info)
	{
		if(this == &info) return *this;
		matrix		= info.matrix;
		U			= info.U;
		lambda	= info.lambda;
		U_inv		= info.U_inv;
		return *this;
	}
	
	const bool operator==(const MatrixInfo<ValueT, Rows, Columns>& info) const
	{
		return matrix == info.matrix; // We assume that the eigenvalues of identical matrices are also identical.
	}
	
	matrix_type matrix;
	matrix_type U;
	matrix_type lambda;
	matrix_type U_inv;
};

class MatrixEigenvaluesCache; // forward declaration

/**
 * @brief Returns a reference to the global MatrixEigenvaluesCache object.
 * @return Reference to the global MatrixEigenvaluesCache object.
 */
MatrixEigenvaluesCache& getEigenvaluesCache();

/**
 * @brief A container which stores matrix dependent information of a
 * matrix. This information contains the matrix itself, its eigenvalues
 * plus U and U^-1. Through this cache it is possible to get the eigenvalues
 * of a given matrix at runtime and compute a matrix exponential faster
 * than with taylor.
 */
class MatrixEigenvaluesCache {
	public:
		typedef std::list<any> info_list;
		typedef info_list::iterator iterator;
		typedef info_list::const_iterator const_iterator;
	
	private:
		MatrixEigenvaluesCache();
		
		friend MatrixEigenvaluesCache& getEigenvaluesCache();
		
		/**
		 * @brief Loads a matrix stored in an array into a fixed_matrix
		 * object.
		 * @param M Destination matrix where the data shall be stored in.
		 * @param array Source array where the matrix data is read from.
		 */
		template<typename ValueT, size_t Rows, size_t Columns>
		void loadMatrixFromArray(fixed_matrix<ValueT, Rows, Columns>& M, const ValueT array[Rows][Columns])
		{
			for(register size_t row = 0; row < Rows; row++)
				for(register size_t col = 0; col < Columns; col++)
					M(row, col) = array[row][col];
		}
		
		/**
		 * @brief Loads the Q_VTML rate matrix for proteins into the
		 * global cache.
		 */
		void loadQVTML();
	
	public:
		~MatrixEigenvaluesCache();
		
		template<typename ValueT, size_t Rows, size_t Columns>
		bool contains(const fixed_matrix<ValueT, Rows, Columns>& t) const
		{
			MatrixInfo<ValueT, Rows, Columns> info(t);
			for(const_iterator i = _list.begin(), end = _list.end(); i != end; i++)
			{
				try
				{
					if(i->type() == typeid(info) && equals(info, any_cast<MatrixInfo<ValueT, Rows, Columns> >(*i)) )
						return true;
				}
				catch(bad_any_cast e) {
					// std::cout << e.what() << std::endl;
				}
			}
			return false;
		}
		
		template<typename ValueT, size_t Rows, size_t Columns>
		bool contains(const MatrixInfo<ValueT, Rows, Columns>& info) const
		{
			return contains(info.matrix);
		}
		
		template<typename ValueT, size_t Rows, size_t Columns>
		bool getMatrixInfo(const fixed_matrix<ValueT, Rows, Columns>& t, MatrixInfo<ValueT, Rows, Columns>& info) const
		{
			info = MatrixInfo<ValueT, Rows, Columns>(t);
			for(const_iterator i = _list.begin(), end = _list.end(); i != end; i++)
			{
				try
				{
					MatrixInfo<ValueT, Rows, Columns> tmp = any_cast<MatrixInfo<ValueT, Rows, Columns> >(*i);
					if(i->type() == typeid(info) && equals(info, tmp))
					{
						info = tmp;
						return true;
					}
				}
				catch(bad_any_cast e) {
					// std::cout << e.what() << std::endl;
				}
			}
			return false;
		}
		
		template<typename ValueT, size_t Rows, size_t Columns>
		bool insert(const MatrixInfo<ValueT, Rows, Columns>& info)
		{
			if(contains(info)) return false;
			_list.push_back(any(info));
			return true;
		}
		
	private:
		template<typename T1, size_t r1, size_t c1, typename T2, size_t r2, size_t c2>
		bool equals(const MatrixInfo<T1, r1, c1>& t1, const MatrixInfo<T2, r2, c2>& t2) const
		{
			return t1 == t2;
		}
		
		info_list		_list;
};



} // namespace profdist

#endif // MATRIX_EIGENVALUES_CACHE_H_
