#include "workspace_tree_view.hpp"
#include "workspace_tree_model.hpp"
#include "debug.hpp"
#include "feature.hpp"
#include "main_window.hpp"
#include "workspace.hpp"

#include <list>

#include <QContextMenuEvent>
#include <QMouseEvent>
#include <QMenu>
#include <QModelIndex>
#include <QFont>
#include <QWidget>
#include <QLineEdit>

namespace gui {
	
	ItemEditDelegate::ItemEditDelegate(QObject* parent)
	: QItemDelegate(parent)
	{}
	
	QWidget* ItemEditDelegate::createEditor(QWidget* parent, const QStyleOptionViewItem& option, const QModelIndex& index) const
	{
		QLineEdit* lineEdit = new QLineEdit(parent);
		lineEdit->setFrame(false);
		return lineEdit;
	}
	
	void ItemEditDelegate::setEditorData(QWidget* editor, const QModelIndex& index) const
	{
		QString data = index.model()->data(index, Qt::DisplayRole).toString();
		QLineEdit* edit = static_cast<QLineEdit*>(editor);
		
		edit->setText(data);
	}
	
	void ItemEditDelegate::setModelData(QWidget* editor, QAbstractItemModel* model, const QModelIndex& index) const
	{
		QLineEdit* edit = static_cast<QLineEdit*>(editor);
		QString data = edit->text();
		model->setData(index, data, Qt::EditRole);
	}
	
	void ItemEditDelegate::updateEditorGeometry(QWidget* editor, const QStyleOptionViewItem& option, const QModelIndex& index) const
	{
		QRect r = option.rect;
		r.setX(r.x() + r.height());
		r.setWidth(r.width() - r.height());
		editor->setGeometry(r);
	}
	
	WorkspaceTreeView::WorkspaceTreeView(QWidget* parent)
	: QTreeView(parent), _context_menu(new QMenu())
	{
		setDragEnabled(false);
		_context_menu->setSeparatorsCollapsible(false);
		
		setEditTriggers(QAbstractItemView::EditKeyPressed);
		
		setItemDelegateForColumn(0, new ItemEditDelegate());
		//setItemDelegate(new ItemEditDelegate());
		
		setIconSize(QSize(24, 24));
		
		// TODO: insert a background image into the workspace background
		//QPalette p = palette(QPalette::Window, QBrush(Q));
		//p.setColorGroup();
	}
	
	void WorkspaceTreeView::contextMenuEvent(QContextMenuEvent* e)
	{
		QModelIndex i = indexAt(e->pos());
		if(i.isValid())
		{
			DataItem* item = static_cast<DataItem*>(i.internalPointer());
			QAction* actionLabel = new QAction(item->icon(), item->name(), _context_menu);
			QFont f = actionLabel->font();
			f.setBold(true);
			actionLabel->setFont(f);
			
			_context_menu->clear();
			_context_menu->addAction(actionLabel);
			_context_menu->addSeparator();
			
			foreach(QAction* a, mainWindow->defaultActions())
			{
				_context_menu->addAction(a);
			}
			
			std::list<Feature*> features = mainWindow->workspace()->getFeaturesForSelectedItem();
			if(!features.empty())
			{
				_context_menu->addSeparator();
				
				QMenu* submenu = _context_menu->addMenu(QIcon(":/images/run.xpm"), tr("&Run"));
				for(std::list<Feature*>::iterator iter = features.begin(), end = features.end(); iter != end; iter++)
				{
					submenu->addAction((*iter)->action());
				}
			}
			
			_context_menu->exec(e->globalPos());
			
			e->accept();
		}
		QTreeView::contextMenuEvent(e);
	}
	
	void WorkspaceTreeView::mousePressEvent(QMouseEvent* e)
	{
		if(indexAt(e->pos()).isValid())
			QTreeView::mousePressEvent(e);
		else
			clearSelection();
		
		emit updateFeatures();
	}
	
}

