#include "workspace.hpp"
#include "main_window.hpp"
#include "workspace_tree_model.hpp"
#include "workspace_tree_view.hpp"
#include "feature.hpp"
#include "data_item.hpp"
#include "task.hpp"
#include "traits.hpp"
#include "bootstrap_operation.hpp"
#include "file_type.hpp"
#include "alignment_data.hpp"
#include "debug.hpp"
#include "plugin_interface.hpp"
#include "ptr_deleter.hpp"
#include "gui_handle.hpp"
#include "settings_dialog.hpp"
#include "viewer.hpp"
#include "distance_operation.hpp"
#include "neighbour_joining_operation.hpp"
#include "consensus_operation.hpp"
#include "profile_neighbour_joining_operation.hpp"
#include "task_manager.hpp"
#include "config.h"
#include "gui_handle.hpp"
#include "gui_handle_data.hpp"

#include "ui_type.h"

#include <QAction>
#include <QString>
#include <QMessageBox>
#include <QFileDialog>
#include <QSettings>
#include <QTreeView>
#include <QDialog>
#include <QFileInfo>
#include <QPluginLoader>
#include <QModelIndex>
#include <QMutexLocker>

namespace gui {
	
	template<typename T>
	struct ObjectDeleter {
		void operator()(T* t)
		{
			if(t)
				delete t;
		}
	};
	
	
	/*
	 * Workspace:
	 */
		
	Workspace::Workspace(MainWindow& mainWindow)
	: QObject(&mainWindow),
	_main_window(&mainWindow),
	_mapping_counter(0),
	_actual_gui_handle_id(0),
	_tree_model(new WorkspaceTreeModel(this)),
	_task_manager(new TaskManager(this))
	{
		connect(&_signal_mapper, SIGNAL(mapped(int)), this, SLOT(triggerOperation(int)));
		connect(_tree_model, SIGNAL(expandNode(const QModelIndex&)), _main_window->workspaceTreeView(), SLOT(expand(const QModelIndex&)));
		
		if(_main_window->workspaceTreeView())
			_main_window->workspaceTreeView()->setModel(_tree_model);
		
		/*
		 * Introduce the basic features to the workspace.
		 * 
		 * install first the pnj operations, so they appear first in the menu
		 */
		installNewFeature(new RnaProfileNeighbourJoiningFeature(), new ProfileNeighbourJoiningOperation<profdist::rna_traits>());
		installNewFeature(new RnaStructureProfileNeighbourJoiningFeature(), new ProfileNeighbourJoiningOperation<profdist::rna_structure_traits>());
		installNewFeature(new ProteinProfileNeighbourJoiningFeature(), new ProfileNeighbourJoiningOperation<profdist::protein_traits>());
		installNewFeature(new RnaBootstrapFeature(), new BootstrapOperation<profdist::rna_traits>());
		installNewFeature(new RnaStructureBootstrapFeature(), new BootstrapOperation<profdist::rna_structure_traits>());
		installNewFeature(new ProteinBootstrapFeature(), new BootstrapOperation<profdist::protein_traits>());
		installNewFeature(new RnaDistanceFeature(), new DistanceOperation<profdist::rna_traits>());
		installNewFeature(new RnaStructureDistanceFeature(), new DistanceOperation<profdist::rna_structure_traits>());
		installNewFeature(new ProteinDistanceFeature(), new DistanceOperation<profdist::protein_traits>());
		installNewFeature(new NeighbourJoiningFeature(), new NeighbourJoiningOperation());
		installNewFeature(new ConsensusFeature(), new ConsensusOperation());
		// TODO: further features can be added
		
		/*
		 * Introduce the basic file types to the workspace.
		 */
		installNewFileType(new Fasta());
		installNewFileType(new Embl());
		installNewFileType(new XFasta());
		
		/*
		 * Install viewers.
		 */
		installViewer(new SimpleViewerFactory());
		installViewer(new ExternViewerFactory());
		
		/*
		 * Install new gui handles
		 */
		installNewGuiHandle(new PnjGuiHandle());
	}
	
	Workspace::~Workspace()
	{
		for_each(_features.begin(), _features.end(), ObjectDeleter<Feature>());
		for_each(_additional_actions.begin(), _additional_actions.end(), ObjectDeleter<QAction>());
		for_each(_data_operations.begin(), _data_operations.end(), ObjectDeleter<OperationInterface>());
		for_each(_file_types.begin(), _file_types.end(), ObjectDeleter<FileType>());
		
		if(_task_manager)
			delete _task_manager;
		
		if(_tree_model)
			delete _tree_model;
		
		GUI_MSG_OBJ_DESTROYED(Workspace);
	}
	
	MainWindow* Workspace::mainWindow()
	{
		return _main_window;
	}
	
	const MainWindow* Workspace::mainWindow() const
	{
		return _main_window;
	}
	
	void Workspace::installNewFeature(Feature* feature, OperationInterface* operation)
	{
		connect(feature->action(), SIGNAL(triggered()), &_signal_mapper, SLOT(map()));
		_signal_mapper.setMapping(feature->action(), _mapping_counter);
		_mapping_counter++;
		_features.push_back(feature);
		_data_operations.push_back(operation);
	}
	
	void Workspace::installNewFileType(FileType* fileType)
	{
		std::list<FileType*>::const_iterator i = std::find(_file_types.begin(), _file_types.end(), fileType);
		if(i != _file_types.end())
			return;
		_file_types.push_back(fileType);
	}
	
	unsigned int Workspace::getNewGuiHandleId()
	{
		return _actual_gui_handle_id++;
	}
	
	int Workspace::installNewGuiHandle(GuiHandle* guiHandle)
	{
		if(!guiHandle || _actual_gui_handle_id == std::numeric_limits<int>::max())
			return -1;
		
		guiHandle->setGuiHandleId(_actual_gui_handle_id);
		_gui_handles.push_back(guiHandle);
		
		return _actual_gui_handle_id++;
	}
	
	void Workspace::useGuiHandle(int handleId, unsigned int taskId,  GuiHandleData* data)
	{
		if(handleId < 0)
			return;
		
		GuiHandle* handle = _gui_handles[handleId]->clone();
		
		if(!handle)
		{
			showErrorMessage(tr("Could not use GuiHandler!"));
			cancelTask(taskId);
			return;
		}
		
		/*
		 * More than one thread can enter this region. To avoid multiple
		 * profile selection windows at a time, we enqueue the rest and compute it
		 * all in the first thread.
		 */
		{
			QMutexLocker locker(&_mutex_use_gui_handle);
			
			GuiHandleEntry entry;
			entry.handle = handle;
			entry.taskId = taskId;
			entry.data = data;
			
			_active_gui_handles.push_back(entry);
			
			/* If a thread is yet working on a gui handle return. */
			if(_active_gui_handles.size() > 1)
				return;
		}
		
		while(!_active_gui_handles.empty())
		{
			GuiHandleEntry e = _active_gui_handles.front();
			if(!mainWindow()->isVisible())
				mainWindow()->onShow();
			e.handle->exec(e.data);
			resumeTask(e.taskId);
			{
				QMutexLocker locker(&_mutex_use_gui_handle);
				_active_gui_handles.pop_front();
			}
			delete e.handle;
		}
		
	}
	
	void Workspace::resumeTask(unsigned int taskId)
	{
		_task_manager->resumeTask(taskId);
	}
	
	void Workspace::cancelTask(unsigned int taskId)
	{
		_task_manager->removeTask(taskId);
	}
	
	bool Workspace::viewerInstalled(unsigned int type)
	{
		return getViewer(type) != 0;
	}
	
	Viewer* Workspace::getViewer(unsigned int type, const QString& name, const QIcon& icon, QWidget* parent)
	{
		if(!_viewers[type])
		{
			return 0;
		}
		return _viewers[type]->clone(name, icon, parent);;
	}
	
	void  Workspace::installViewer(ViewerFactory* factory)
	{
		if(!factory)
		{
			return;
		}
		
		foreach(unsigned int type, factory->types())
		{
			_viewers[type] = factory->getPrototype();
		}
	}
	
	const std::list<FileType*> Workspace::getFileTypes(const std::string& suffix)
	{
		return getFileTypes(QString(suffix.c_str()));
	}
	
	const std::list<FileType*> Workspace::getFileTypes(const QString& suffix)
	{
		std::list<FileType*> types;
		for(std::list<FileType*>::iterator i = _file_types.begin(),
			e = _file_types.end(); i != e; i++)
			if((*i)->containsSuffix(suffix))
				types.push_back(*i);
		
		return types;
	}
	
	QString Workspace::getFileTypeFilter() const
	{
		QString filter;
		
		for(std::list<FileType*>::const_iterator i = _file_types.begin(),
			e = _file_types.end(); i != e; i++)
		{
			filter += (*i)->fileDialogFilter() + ";;";
		}
		
		filter += "All Files (*)";
		
		return filter;
	}
	
	DataItem* Workspace::getSelectedItem()
	{
		DataItem* item = 0;
		
		if(_main_window->workspaceTreeView())
		{
			QModelIndexList l = _main_window->workspaceTreeView()->selectionModel()->selectedIndexes();
			
			if(!l.empty())
			{
				QModelIndex index = l.front();
				item = static_cast<DataItem*>(index.internalPointer());
			}
		}
		
		return item;
	}
	
	bool Workspace::hasRunningTasks()
	{
		return _task_manager->hasRunningTasks();
	}
	
	std::list<Feature*> Workspace::getFeaturesForSelectedItem()
	{
		DataItem* item = getSelectedItem();
		
		if(!item)
			return std::list<Feature*>();
		
		std::list<Feature*> featureList;
		
		foreach(Feature* f, _features)
		{
			if(f->isAvailable(item->type()))
				featureList.push_back(f);
		}
		
		return featureList;
	}
	
	void Workspace::loadPlugins()
	{
		
		QDir pluginsDir(PROFDIST_PLUGINS_DIR);
		
		foreach(QString fileName, pluginsDir.entryList(QStringList() << "lib*"))
		{
			QPluginLoader loader(pluginsDir.absoluteFilePath(fileName));
			QObject* object = loader.instance();
			
			if(object)
			{
				PluginInterface* plugin = qobject_cast<PluginInterface*>(object);
				
				if(plugin)
				{
					plugin->installPlugin(this);
				}
			}
		}
	}
	
	void Workspace::addDataItem(DataItem* newItem, DataItem* parentItem)
	{
		if(newItem && _tree_model)
			_tree_model->addNode(newItem, parentItem);
	}
	
	void Workspace::removeDataItem(DataItem* item)
	{
		if(item && _tree_model)
			_tree_model->removeNode(item);
	}
	
	void Workspace::triggerOperation(int index)
	{
		if(index < 0 && index < _data_operations.size())
			return;
		
		OperationInterface* op = _data_operations[index]->clone();
		DataItem* item = getSelectedItem();
		
		if(!op->setDataItem(item))
		{
			showErrorMessage(tr("Could not start operation.\nIncorrect data type"));
			return;
		}
		
		_task_manager->addTask(op);
	}
	
	void Workspace::removeTask(Task* task)
	{
		if(!task)
			return;
		
		_task_manager->removeTask(task->taskId());
	}
	
	void Workspace::open()
	{
		QSettings settings;
		QString path = settings.value("settings/MainWindow/open_path").toString();
		if(path.isNull())
#ifdef Q_OS_WIN32
			path = QCoreApplication::applicationDirPath();
#else
			path = PROFDIST_DATA_DIR "/data";
#endif
		
		QString fileName = QFileDialog::getOpenFileName(
			_main_window,
			tr("Open file"),
			path,
			getFileTypeFilter());
		
		if(fileName.isNull())
			return;
		
		QDir dir(fileName);
		dir.cdUp();
		settings.setValue("settings/MainWindow/open_path", dir.canonicalPath());
		
		QFileInfo info(fileName);
		const std::list<FileType*> fileTypes = getFileTypes(info.suffix());
		
		const FileType* fileType = 0;
		
		if(fileTypes.empty())
			fileType = chooseFileType(_file_types, fileName);
		else if(fileTypes.size() == 1)
			fileType = fileTypes.front();
		else
			fileType = chooseFileType(fileTypes, fileName);
		
		if(!fileType)
		{
			showErrorMessage(tr("Could not determine the file type."));
			return;
		}
		
		DataItem* item = 0;
		
		try
		{
			item = fileType->parseFile(fileName);
		}
		catch(std::runtime_error& e)
		{
			showErrorMessage(tr("Error while parsing file '") +
				fileName + tr("':\n") + QString(e.what()));
			return;
		}
		
		if(item)
		{
			_tree_model->addNode(item);
			_main_window->updateFeatures();
		}
		else
		{
			showErrorMessage(tr("Error while adding data item to workspace."));
		}
	}
	
	void Workspace::remove()
	{
		DataItem* item = getSelectedItem();
		
		if(item)
		{
			_tree_model->removeNode(item);
			_main_window->updateFeatures();
		}
	}
	
	void Workspace::rename()
	{
		QModelIndexList indexes = mainWindow()->workspaceTreeView()->selectionModel()->selectedIndexes();
		if(indexes.empty())
			return;
		QModelIndex index = indexes.front();
		if(!index.isValid())
			return;
		mainWindow()->workspaceTreeView()->setCurrentIndex(index);
		mainWindow()->workspaceTreeView()->edit(index);
	}
	
	void Workspace::saveAs()
	{
		DataItem* item = getSelectedItem();
		
		if(!item)
			return;
		
		QSettings settings;
		
		QString fileName = QFileDialog::getSaveFileName(
			_main_window,
			tr("Save as"),
			settings.value("gui/MainWindow/save_path").toString());
		
		if(fileName.isNull())
			return;
		
		QDir dir(fileName);
		dir.cdUp();
		settings.setValue("gui/MainWindow/save_path", dir.canonicalPath());
		
		profdist::FileType fileType;
		
		QFileInfo info(fileName);
		if(info.suffix() == "embl")
			fileType = profdist::Embl;
		else
			fileType = profdist::Fasta;
		
		try
		{
			std::ofstream out(fileName.toStdString().c_str());
			if(!out)
			{
				throw std::runtime_error(std::string("Could not open file ") + fileName.toStdString());
			}
			if(!item->writeData(out, fileType))
			{
				throw std::runtime_error("Error while writing to file");
			}
		}
		catch(std::runtime_error& e)
		{
			showErrorMessage(tr("Could not save data:\n") + QString(e.what()));
		}
	}
	
	void Workspace::view()
	{
		DataItem* item = getSelectedItem();
		
		if(!item)
		{
			return;
		}
		
		if(!item->viewer())
		{
			if(!viewerInstalled(item->type()))
			{
				mainWindow()->showMessage(tr("No viewer registered for that data type"), tr("Information"), MainWindow::MT_INFORMATION);
				return;
			}
			else
			{
				Viewer* v = getViewer(item->type(), item->name(), item->icon(), mainWindow());
				v->setData(item->dataToString());
				item->setViewer(v);
			}
		}
		
		item->viewer()->show();
	}
	
	void Workspace::settings()
	{
		SettingsDialog dialog(mainWindow());
		dialog.exec();
	}
	
	void Workspace::showErrorMessage(const std::string& message)
	{
		showErrorMessage(QString(message.c_str()));
	}
	
	void Workspace::showErrorMessage(const QString& message)
	{
		QMessageBox::critical(_main_window, tr("Error"), message);
	}
	
	void Workspace::balloonMessage(const QString& message)
	{
		_main_window->ballonMessage(message);
	}
	
	void Workspace::balloonMessage(const QString& title, const QString& message)
	{
		_main_window->ballonMessage(title, message);
	}
	
	const FileType* Workspace::chooseFileType(const std::list<FileType*>& fileTypes,
		const QString& fileName)
	{
		QDialog dia(_main_window);
		Ui::FileTypeDialog dialog;
		dialog.setupUi(&dia);
		
		QStringList types;
		foreach(FileType* ft, fileTypes)
		{
			types.push_back(ft->name());
		}
		
		dialog.FileType->addItems(types);
		
		QSettings settings;
		
		int index = dialog.FileType->findText( settings.value( "gui/MainWindow/ask_file_type" ).toString() );
		if( index == -1 ) index = 0;
		
		dialog.FileType->setCurrentIndex( index );
		dialog.FileLabel->setText( fileName );
		dia.exec();
		
		settings.setValue( "gui/MainWindow/ask_file_type", dialog.FileType->currentText() );
		
		index = dialog.FileType->currentIndex();
		
		std::list<FileType*>::const_iterator it = fileTypes.begin();
		for( size_t i = 0; i != index; ++i,++it );
		
		return *it;
	}
	
}

