#include "viewer.hpp"
#include <QTextEdit>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QFont>
#include <QLabel>
#include <QFrame>
#include <QPixmap>
#include <QPushButton>
#include <QProcess>
#include <QSettings>
#include <QRegExp>
#include <QMessageBox>

#include "main_window.hpp"
#include "data_type.hpp"

#include "tree_selection_dialog.hpp"

namespace gui {
	
	/*
	 * SimpleViewer:
	 */
	SimpleViewer::SimpleViewer(const QString& name, const QIcon& icon, QWidget* parent)
	: _text_edit(new QTextEdit())
	{
		_text_edit->setLineWrapMode(QTextEdit::NoWrap);
		_text_edit->setReadOnly(true);
		QFont f("Courier", 10);
		_text_edit->setCurrentFont(f);
		
		setWindowTitle(name + tr(" - Viewer"));
		
		QVBoxLayout* vlayout = new QVBoxLayout();
		QHBoxLayout* hlayout = new QHBoxLayout();
		QHBoxLayout* hlayout2 = new QHBoxLayout();
		QLabel* label_icon = new QLabel();
		QLabel* label_name = new QLabel();
		QFrame* frame_line = new QFrame(this);
		QPushButton* closeButton = new QPushButton(tr("Close"));
		
		connect(closeButton, SIGNAL(clicked()), this, SLOT(close()));
		
		label_icon->setPixmap(icon.pixmap(QSize(35,35)));
		label_icon->setMaximumWidth(40);
		
		label_name->setText(name);
		QFont f2(label_name->font());
		f2.setPointSize(14);
		label_name->setFont(f2);
		
		frame_line->setFrameStyle(QFrame::Sunken | QFrame::HLine);
		
		hlayout->addWidget(label_icon);
		hlayout->addWidget(label_name);
		
		hlayout2->addStretch();
		hlayout2->addWidget(closeButton);
		
		vlayout->addLayout(hlayout);
		vlayout->addWidget(frame_line);
		vlayout->addWidget(_text_edit);
		vlayout->addLayout(hlayout2);
		setLayout(vlayout);
		resize(700, 500);
		if(parent)
		{
			move(parent->x() + parent->width()/2 - width()/2, parent->y() + parent->height()/2 - height()/2);
		}
	}
	
	Viewer* SimpleViewer::clone(const QString& name, const QIcon& icon, QWidget* parent) const
	{
		return new SimpleViewer(name, icon, parent);
	}
	
	void SimpleViewer::setData(const QString& data)
	{
		_text_edit->setText(data);
	}
	
	void SimpleViewer::show(bool modal)
	{
		QDialog::show();
	}
	
	/*
	 * SimpleViewerFactory:
	 */
	Viewer* SimpleViewerFactory::getPrototype() const
	{
		return new SimpleViewer();
	}
	
	QList<unsigned int> SimpleViewerFactory::types() const
	{
		return QList<unsigned int>() << DataType::SingleRnaAlignment
				<< DataType::SingleRnaStructureAlignment
				<< DataType::SingleProteinAlignment
				<< DataType::RnaBootstrap
				<< DataType::RnaStructureBootstrap
				<< DataType::ProteinBootstrap
				<< DataType::DistanceMatrix;
	}
	
	/*
	 * ExternViewer:
	 */
	ExternViewer::ExternViewer(const QString& name, const QIcon& icon, QWidget* parent)
	: _tmp_file("profdist_extern_viewer_XXXXXX.tre"), _tmp_file_name(""), _process(0), _data()
	{}
	
	ExternViewer::~ExternViewer()
	{
		if(_process)
		{
			_process->kill();
			_process->waitForFinished(200);
			delete _process;
		}
	}
	
	Viewer* ExternViewer::clone(const QString& name, const QIcon& icon, QWidget* parent) const
	{
		return new ExternViewer(name, icon, parent);
	}
	
	void ExternViewer::setData(const QString& data)
	{
		if(!_tmp_file.open())
			return;
		_data = data;
		_tmp_file_name = _tmp_file.fileName();
		_tmp_file.write(data.toUtf8());
		_tmp_file.close();
	}
	
	void ExternViewer::show(bool modal)
	{
		QStringList trees = _data.split(";", QString::SkipEmptyParts);
		
		
		QStringList::iterator iter = std::remove(trees.begin(), trees.end(), QString("\n"));
		if(iter != trees.end())
			trees.erase(iter);
		//std::remove(trees.begin(), trees.end(), QString("\n"));
		
		for(QStringList::iterator i = trees.begin(), e = trees.end(); i != e; i++)
		{
			(*i).replace(QRegExp("[\n\r]+"), "");
			(*i) += ";";
		}
		
		QSettings settings;
		
		int viewer = settings.value("tree_viewer/selected").toInt();
		
		if(viewer == 0)
		{
			QMessageBox::information(mainWindow, QObject::tr("No viewer"), QObject::tr("No viewer was specified in the settings."));
			return;
		}
		
		if(_tmp_file_name.isNull())
		{
			mainWindow->showMessage(QObject::tr("Could not open extern viewer because\ntemporary file name is empty."));
			return;
		}
		
		switch(viewer)
		{
			case 1: // TreeView
			{
				QString viewerBinPath = settings.value("tree_viewer/tv").toString();
				if(viewerBinPath == "")
				{
					QMessageBox::information(mainWindow, QObject::tr("No viewer"), QObject::tr("No path to the viewer's executable specified.\nPlease check the settings."));
					return;
				}
				if(_process)
					delete _process;
				_process = 0;
				_process = new QProcess(mainWindow);
				_process->start(viewerBinPath, QStringList() << _tmp_file_name);
				break;
			}
			case 2: // NjPlot
			{
				if(settings.value("tree_viewer/njplot").toString() == "")
				{
					QMessageBox::information(mainWindow, QObject::tr("No viewer"), QObject::tr("No path to the viewer's executable specified.\nPlease check the settings."));
					return;
				}
				if(trees.size() > 1)
				{
					TreeSelectionDialog dialog(trees, mainWindow);
					if(!modal)
					{
						dialog.show();
						dialog.raise();
						dialog.activateWindow();
					}
					else
					{
						dialog.exec();
					}
				}
				else {
					QString viewerBinPath = settings.value("tree_viewer/njplot").toString();
					if(viewerBinPath == "")
					{
						QMessageBox::information(mainWindow, QObject::tr("No viewer"), QObject::tr("No path to the viewer's executable specified.\nPlease check the settings."));
						return;
					}
					if(_process)
						delete _process;
					_process = 0;
					_process = new QProcess(mainWindow);
					_process->start(viewerBinPath, QStringList() << _tmp_file_name);
					}
				break;
			}
		}
		
		/*
		QStringList viewers = settings.value("settings/tree_viewer").toStringList();
			
		if(viewers.empty())
		{
			mainWindow->showMessage(QObject::tr("Could not open tree in extern viewer because\nno viewer were specified in the settings."));
			return;
		}
		
		if(_tmp_file_name.isNull())
		{
			mainWindow->showMessage(QObject::tr("Could not open extern viewer because\ntemporary file name is empty."));
			return;
		}
		
		if(trees.size() == 1 ||
			(trees.size() > 1 &&
				QMessageBox::Yes == QMessageBox::question(mainWindow,
					QObject::tr("Multiple trees"),
					QObject::tr("Your actual tree viewer is\n") + viewers.front() + QObject::tr(".\nDo you want to open all trees in that viewer?"),
					QMessageBox::Yes | QMessageBox::No,
					QMessageBox::No)))
		{
			_process = new QProcess(mainWindow);
			_process->start(viewers.front(), QStringList() << _tmp_file_name);
		}
		else
		{
			TreeSelectionDialog dialog(trees, mainWindow);
			dialog.exec();
		}
		*/
		
	}
	
	/*
	 * ExternViewerFactory:
	 */
	Viewer* ExternViewerFactory::getPrototype() const
	{
		return new ExternViewer();
	}
	
	QList<unsigned int> ExternViewerFactory::types() const
	{
		return QList<unsigned int>() << DataType::NjTree
				<< DataType::ConsensusTree
				<< DataType::PnjTree;
	}
	
}
