#include "task_container_widget.hpp"
#include "task.hpp"
#include "debug.hpp"
#include <QLabel>
#include <QProgressBar>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QToolButton>
#include <QMessageBox>
#include <QResizeEvent>

namespace gui {
	
	TaskProgressWidget::TaskProgressWidget(TaskContainerWidget* container)
	: _label(new QLabel()),
	_progress_bar(new QProgressBar()),
	_cancel_button(new QToolButton()),
	_resume_button(new QToolButton()),
	_container(container)
	{
		_progress_bar->setValue(0);
		
		QVBoxLayout* vlayout = new QVBoxLayout();
		QHBoxLayout* hlayout = new QHBoxLayout();
		
		_cancel_button->setIcon(QIcon(":/images/fileclose.png"));
		connect(_cancel_button, SIGNAL(clicked()), this, SLOT(cancelButtonClicked()));
		
		_resume_button->setIcon(QIcon(":/images/play.png"));
		_resume_button->setEnabled(false);
		connect(_resume_button, SIGNAL(clicked()), this, SIGNAL(resumeTask()));
		//connect(_resume_button, SIGNAL(clicked()), _resume_button, SLOT(disable()));
		
		hlayout->addWidget(_progress_bar);
		hlayout->addWidget(_cancel_button);
		hlayout->addWidget(_resume_button);
		
		vlayout->addWidget(_label);
		vlayout->addLayout(hlayout);
		
		setLayout(vlayout);
	}
	
	TaskProgressWidget::~TaskProgressWidget()
	{
		//if(_container)
		//	_container->removeTaskProgressWidget(this);
	}
	
	void TaskProgressWidget::taskPaused()
	{
		_resume_button->setEnabled(true);
	}
	
	void TaskProgressWidget::deleteProgressWidget()
	{
		if(_container)
			_container->removeTaskProgressWidget(this);
		delete this;
	}
	
	void TaskProgressWidget::cancelButtonClicked()
	{
		QMessageBox dialog(
			QMessageBox::Question,
			tr("Cancel Task"),
			tr("Do you really want to cancel the task?"),
			QMessageBox::Yes | QMessageBox::No,
			this);
		
		connect(this, SIGNAL(closeDialog()), &dialog, SLOT(reject()));
		
		if(QMessageBox::Yes == dialog.exec())
		{
			emit cancelTask();
			_cancel_button->setEnabled(false);
			updateProgress(tr(" [CANCELING]"));
		}
	}
	
	void TaskProgressWidget::setMaximumValue(unsigned int maximumValue)
	{
		_progress_bar->setRange(0, maximumValue);
	}
	
	void TaskProgressWidget::updateProgress(unsigned int value)
	{
		_progress_bar->setValue(value);
		if(value == _progress_bar->maximum() - 1)
			emit closeDialog();
	}
	
	void TaskProgressWidget::updateProgress(const QString message)
	{
		_label->setText(message);
	}
	
	/*
	 * TaskContainerWidget:
	 */
	
	TaskContainerWidget::TaskContainerWidget(QWidget* parent)
	: QScrollArea(parent), _layout(new QVBoxLayout()), _w(new QWidget())
	{
		_w->resize(width()-1, 0);
		_w->setLayout(_layout);
		_layout->setMargin(0);
		_layout->setSpacing(0);
		setWidget(_w);
	}
	
	void TaskContainerWidget::trackProgressOfTask(Task* task)
	{
		TaskProgressWidget* widget = new TaskProgressWidget(this);
		
		QObject::connect(task, SIGNAL(sigUpdateProgress(unsigned int)), widget, SLOT(updateProgress(unsigned int)));
		QObject::connect(task, SIGNAL(sigUpdateProgress(const QString&)), widget, SLOT(updateProgress(const QString&)));
		QObject::connect(task, SIGNAL(sigSetMaximumValue(unsigned int)), widget, SLOT(setMaximumValue(unsigned int)));
		QObject::connect(task, SIGNAL(taskFinished(unsigned int)), widget, SLOT(deleteProgressWidget()));
		QObject::connect(widget, SIGNAL(cancelTask()), task, SLOT(cancel()));
		
		addTaskProgressWidget(widget);
	}
	
	void TaskContainerWidget::removeTaskProgressWidget(TaskProgressWidget* widget)
	{
		if(!widget)
			return ;
		widget->setParent(0);
		_layout->removeWidget(widget);
		_w->resize(_w->width(), _w->height() - widget->height());
	}
	
	void TaskContainerWidget::addTaskProgressWidget(TaskProgressWidget* widget)
	{
		if(!widget)
			return;
		_layout->addWidget(widget);
		_w->resize(_w->width(), _w->height() + widget->sizeHint().height());
	}
	
	void TaskContainerWidget::resizeEvent(QResizeEvent* e)
	{
		_w->resize(e->size().width(), _w->height());
		QScrollArea::resizeEvent(e);
	}
	
}
