#include "task.hpp"
#include "data_item.hpp"
#include "operation_interface.hpp"
#include "main_window.hpp"
#include "workspace.hpp"
#include "debug.hpp"
#include <algorithm>
#include <QString>

namespace gui {
	
	/*
	 * Task:
	 */
	unsigned int newTaskId()
	{
		static unsigned int actTaskId = 1;
		return actTaskId++;
	}
	
	Task::Task(OperationInterface* operation)
	: _task_id(newTaskId()), _is_canceled(false), _operation(operation), _ended_regular(false)
	{
		if(_operation)
			_operation->setTask(this);
	}
	
	Task::~Task()
	{
		cancel();
		if(!wait(TaskCancelTimeout) && isRunning())
		{
			// if the thread computes large operation with long update intervals
			// it can happen that the application has to wait long time until
			// an update event occurs and the thread returns from the run
			// method.
			terminate();
			wait();
			emit taskFinished(_task_id);
		}
		
		GUI_MSG_OBJ_DESTROYED(Task);
	}
	
	unsigned int Task::taskId() const
	{
		return _task_id;
	}
	
	void Task::start()
	{
		QThread::start(QThread::NormalPriority);
	}
	
	void Task::cancel()
	{
		_is_canceled = true;
		if(_operation)
			_operation->cancel();
	}
	
	bool Task::wait(unsigned long time)
	{
		return QThread::wait(time);
	}
	
	bool Task::isCanceled() const
	{
		return _operation ? _operation->isCanceled() : _is_canceled;
	}
	
	boost::shared_ptr<OperationInterface> Task::operation()
	{
		return _operation;
	}
	
	void Task::pause(int guiHandleId, GuiHandleData* data)
	{
		_mutex_condition.lock();
		emit sigUseGuiHandle(guiHandleId, taskId(), data);
		_pause_condition.wait(&_mutex_condition);
		_mutex_condition.unlock();
	}
	
	void Task::setMaximumValue(size_t maximumValue)
	{
		emit sigSetMaximumValue(maximumValue);
	}
	
	void Task::updateProgress(size_t value)
	{
		emit sigUpdateProgress(value);
	}
	
	void Task::updateProgress(const QString& message)
	{
		emit sigUpdateProgress(message);
	}
	
	void Task::addDataItem(DataItem* newNode, DataItem* parentNode)
	{
		emit sigAddDataItem(newNode, parentNode);
	}
	
	void Task::showErrorMessage(const QString& message)
	{
		emit sigShowErrorMessage(message);
	}
	
	void Task::resume()
	{
		_pause_condition.wakeOne();
	}
	
	void Task::run()
	{
		if(_operation)
			_operation->execute();
		emit taskFinished(_task_id);
	}
	
}

