#ifndef PROFDIST_QTGUI_PROFILE_NEIGHBOUR_JOINING_OPERATION_HPP_
#define PROFDIST_QTGUI_PROFILE_NEIGHBOUR_JOINING_OPERATION_HPP_

#include "operation_interface.hpp"
#include "operation_accessors.hpp"
#include "data_item.hpp"
#include "profile_selection_dialog.hpp"
#include "types.h"
#include "progress_update.h"
#include "tree.h"
#include "profile.h"
#include "classification_data.hpp"
#include "runtime_settings.hpp"
#include "bionj_clean.h"
#include "profile_neighbour_joining_data.hpp"
#include "main_window.hpp"
#include "debug.hpp"
#include "bootstrap.h"
#include "distance.h"
#include "gui_handle.hpp"
#include "gui_handle_data.hpp"
#include "viewer.hpp"

#include <algorithm>

namespace gui {
	
	template<typename Traits>
	class ProfileNeighbourJoiningOperation : public OperationInterface {
		public:
									ProfileNeighbourJoiningOperation();
									~ProfileNeighbourJoiningOperation();
			void					execute();
			OperationInterface*		clone();
			bool					setDataItem(DataItem* item);
			bool					preExecution();
			
		private:
			bool					loadRateMatrix(const QString& fileName, typename Traits::rate_matrix& matrix);
			bool					startPnjMainLoop(profdist::Profile& profile,
										std::vector<std::string> const& sequence_names,
										tree_types::profile_map& profiles_found,
										typename Traits::rate_matrix const& Q,
										std::ostream& out);
			
		private:
			DataForPnj<Traits>		*_data_for_pnj;
			size_t					_num_bootstraps;
			profdist::CorrectionModel	_correction;
			QString					_filename_rate_matrix;
			int						_pnj_method;
			size_t					_threshold_minimum_bootstrap;
			size_t					_threshold_identity;
			profdist::Profile		*_profile;
			ProfileSelectionDialog::profile_list_t				_profiles;
			ProfileSelectionDialog::profile_names_t				_profile_names;
			ProfileSelectionDialog::profile_sequence_names_t	_profile_sequence_names;
			
	};
	
	template<typename Traits>
	ProfileNeighbourJoiningOperation<Traits>::ProfileNeighbourJoiningOperation()
	: _profile(0)
	{}
	
	template<typename Traits>
	ProfileNeighbourJoiningOperation<Traits>::~ProfileNeighbourJoiningOperation()
	{
		if(_profile)
			delete _profile;
		
		GUI_MSG_OBJ_DESTROYED(ProfileNeighbourJoiningOperation);
	}
	
	template<typename Traits>
	void ProfileNeighbourJoiningOperation<Traits>::execute()
	{
		if(!_profile)
		{
			showErrorMessage(QObject::tr("No profile"));
			return;
		}
		size_t maximumValue = std::max(size_t(1), _num_bootstraps);
		setMaximumValue(maximumValue + 3);
		updateProgress(0);
		updateProgress(QObject::tr("Profile Neighbour Joining (") + _data_for_pnj->name() + ")");
		
		tree_types::profile_map profiles_found;
		std::size_t num_profiles = _profile->get_num_profiles();
		RuntimeSettings settings;
		profdist::ProgressSink sink;
		profdist::distance_matrix matrix( num_profiles, num_profiles, 0.0 );
		typename Traits::rate_matrix Q(0);
		std::ostringstream out;
		
		if(_correction >= profdist::GTR && !loadRateMatrix(_filename_rate_matrix, Q))
			return;
		
		try
		{
			profdist::compute_distance<Traits>( *_profile, matrix, Q, _correction, profdist::Eigenvalue, profdist::NewtonMethod);
			bionj( matrix, _profile_names, out, sink );
			
			std::srand(std::time(NULL));
			std::size_t mini_step = 0;
			CountedSets cons;
			
			if(_num_bootstraps)
			{
				for(size_t i = 0; i < _num_bootstraps; i++)
				{
					if(isCanceled())
						return;
					profdist::Profile profile_b(_profile->get_num_sites(), _profile->get_num_profiles());
					profdist::bootstrap(*_profile, profile_b);
					
					profdist::compute_distance<Traits>(profile_b, matrix, Q, _correction, profdist::Eigenvalue, profdist::NewtonMethod);
					profdist::bionj(matrix, cons, sink);
					updateProgress(++mini_step);
				}
			}
			else
			{
				profdist::compute_distance<Traits>(*_profile, matrix, Q, _correction, profdist::Eigenvalue, profdist::NewtonMethod);
				profdist::bionj(matrix, cons, sink);
				updateProgress(++mini_step);
			}
			
			/*
			* Build consensus tree
			*/
			updateProgress(QObject::tr("Creating consensus of all trees (") + _data_for_pnj->name() + ")");
			cons.consense(num_profiles);
			updateProgress(++mini_step);
			
			/*
			* Rebuilding tree
			*/
			updateProgress(QObject::tr("Rebuilding trees (") + _data_for_pnj->name() + ")");
			Tree tree(cons, num_profiles, _num_bootstraps);
			updateProgress(++mini_step);
			
			cons.clear();
			
			tree.print(out, _profile_names, _num_bootstraps);
			out << std::endl << std::flush;
			
			std::ostringstream out_tmp;
			tree.print(out_tmp, _profile_names, _num_bootstraps);
			out_tmp << std::endl << std::flush;
			
			/*
			* Searching for new profiles
			*/
			updateProgress(QObject::tr("Searching for new profiles (") + _data_for_pnj->name() + ")");
			profdist::profile_set profiles;
			if(_pnj_method < ProfileSelectionDialog::Automatic)
			{
				PnjGuiHandleData data;
				data.icon = _data_for_pnj->icon();
				data.alignmentName = _data_for_pnj->name();
				data.tree = out_tmp.str();
				pause(PnjGuiHandle::getGuiHandleId(), &data);
				profiles = data.profiles;
			}
			
			size_t threshold = static_cast<size_t>(static_cast<float>(_threshold_minimum_bootstrap * _num_bootstraps) / 100.0);
			if(_threshold_minimum_bootstrap > 100)
				threshold = _threshold_minimum_bootstrap + 1;
			
			profdist::identical_seq_set ident;
			_profile->get_identical_sequences(ident, static_cast<float>(_threshold_identity) / 100.0);
			tree.find_profile_first(profiles_found, profiles, ident, threshold, _pnj_method);
			
			updateProgress(QObject::tr("Task finished (") + _data_for_pnj->name() + ")");
			updateProgress(++mini_step);
			
			if(profiles_found.size() != num_profiles)
			{
				_profile->refine(profiles_found);
				std::vector<std::string> sequence_names(_data_for_pnj->getSequenceNames());
				if(!startPnjMainLoop(*_profile, _profile_names, profiles_found, Q, out))
					return;
			}
			
			out.flush();
			
			// TODO: add code to add the distance matrices and several temporary
			// stuff computed during pnj in the tree.
			ProfileNeighbourJoiningData* node = new ProfileNeighbourJoiningData(_data_for_pnj->name() + "_pnj", out.str());
			addDataItem(node, _data_for_pnj);
		}
		catch(std::runtime_error& e)
		{
			GUI_MSG_N("Profile Neighbour Joining error:" << std::endl << e.what());
			showErrorMessage(QObject::tr("Profile Neighbour Joining error:\n") + e.what());
			return;
		}
		catch(std::exception& e)
		{
			GUI_MSG_N("Profile Neighbour Joining error:" << std::endl << e.what());
			showErrorMessage(QObject::tr("Profile Neighbour Joining error:\n") + e.what());
			return;
		}
		
		_data_for_pnj->decrementUseCount();
	}
	
	template<typename Traits>
	OperationInterface* ProfileNeighbourJoiningOperation<Traits>::clone()
	{
		return new ProfileNeighbourJoiningOperation();
	}
	
	template<typename Traits>
	bool ProfileNeighbourJoiningOperation<Traits>::setDataItem(DataItem* item)
	{
		if(!item)
			return false;
		
		_data_for_pnj = dynamic_cast<DataForPnj<Traits>*>(item);
		if(!_data_for_pnj)
			return false;
		
		_data_for_pnj->incrementUseCount();
		return true;
	}
	
	/*
	 * This method handles the template parameters profdist::rna_structure_traits
	 * and profdist::protein_traits. In this cases it doesn't provide the kimura-2-parameter
	 * correction model. In the implementation file a specialized method is
	 * provided for the profdist::rna_traits case where we can provide a
	 * kimura-2-parameter correction model.
	 */
	template<typename Traits>
	bool ProfileNeighbourJoiningOperation<Traits>::preExecution()
	{
		RuntimeSettings settings;
		
		ProfileSelectionDialog dialog(_data_for_pnj->getSequenceNames(), _data_for_pnj->getClassificationData(), true, mainWindow);
		dialog._label_icon->setPixmap(_data_for_pnj->icon().pixmap(dialog._label_icon->size()));
		dialog._label_sequence_name->setText(_data_for_pnj->name());
		//dialog._combo_box_generation_method->setCurrentIndex(settings.profile_generation);
		dialog._spin_box_minimal_bootstrap->setValue(settings.bootstrap_threshold);
		dialog._spin_box_identity->setValue(settings.identity_threshold);
		
		if(QDialog::Rejected == dialog.exec())
			return false;
		
		_num_bootstraps = static_cast<size_t>(dialog._spin_box_num_bootstraps->value());
		_correction = dialog.correctionModel();
		_filename_rate_matrix = dialog._line_edit_rate_matrix->text();
		_pnj_method =  ProfileSelectionDialog::PnjMethod(dialog._combo_box_generation_method->currentIndex());
		_threshold_minimum_bootstrap = static_cast<size_t>(dialog._spin_box_minimal_bootstrap->value());
		_threshold_identity = static_cast<size_t>(dialog._spin_box_identity->value()); 
		
		dialog.getProfiles(_profiles, _profile_names, _profile_sequence_names);
		
		// Initialize the profile
		_profile = new profdist::Profile(_data_for_pnj->alignCode(), _profiles);
		
		return true;
	}
	
#ifdef Q_OS_WIN32
	template<>
	bool ProfileNeighbourJoiningOperation<profdist::rna_traits>::preExecution();
#endif
	
	template<typename Traits>
	bool ProfileNeighbourJoiningOperation<Traits>::loadRateMatrix(const QString& fileName, typename Traits::rate_matrix& matrix)
	{
		if(fileName.isNull())
			return false;
		
		ifstream infile(fileName.toStdString().c_str());
		if(!infile)
		{
			GUI_MSG_N("Profile Neighbour Joining error: could not open file to read rate matrix");
			showErrorMessage(QObject::tr("Profile Neighbour Joining error:\nCould not open file '") + fileName + QObject::tr("' to read rate matrix."));
			return false;
		}
		if(!profdist::read_rate_matrix<Traits>(infile, matrix) && !infile.fail())
		{
			GUI_MSG_N("error while loading rate matrix");
			showErrorMessage(QObject::tr("Profile Neighbour Joining error:\nCould not read rate matrix from file '") + fileName + "'");
			return false;
		}
		return true;
	}
	
	template<typename Traits>
	bool ProfileNeighbourJoiningOperation<Traits>::startPnjMainLoop(
		profdist::Profile& profile,
		std::vector<std::string> const& sequence_names,
		tree_types::profile_map& profiles_found,
		typename Traits::rate_matrix const& Q,
		std::ostream& out)
	{
		try
		{
			size_t max = std::max(static_cast<long int>(_num_bootstraps), 1L) + 4;
			size_t num_profiles = profile.get_num_profiles();
			size_t num_new_profiles = 1;
			size_t step = 2;
			RuntimeSettings settings;
			
			/*
			* Main loop
			*/
			while( num_new_profiles && num_profiles > 3)
			{
				if(isCanceled())
					return false;
				
				size_t mini_step = 0;
				QString message = QObject::tr("Profile Neighbour Joining (step ")
						+ QString().setNum(step++)
						+ QObject::tr(")");
				
				QString message_1 = QObject::tr(": bootstrap, distance, bionj");
				
				profdist::distance_matrix matrix(num_profiles, num_profiles, 0.0);
				CountedSets split_sets;
				
				if(_num_bootstraps)
				{
					for(int i = 0; i < _num_bootstraps; i++)
					{
						if(isCanceled())
							return false;
						
						updateProgress(message + message_1);
						
						profdist::Profile other_profile(
								_profile->get_num_sites(),
								_profile->get_num_profiles());
						
						profdist::bootstrap(*_profile, other_profile);
						
						
						profdist::compute_distance<Traits>(
								other_profile,
								matrix,
								Q,
								_correction,
								profdist::Eigenvalue,
								profdist::NewtonMethod
						);
						
						profdist::ProgressSink sink;
						profdist::bionj(matrix, split_sets, sink);
						
						updateProgress(++mini_step);
					}
				}
				else
				{
					updateProgress(message + message_1);
					
					profdist::compute_distance<Traits>(
							*_profile,
							matrix,
							Q,
							_correction,
							profdist::Eigenvalue,
							profdist::NewtonMethod);
				
					profdist::ProgressSink sink;
					profdist::bionj(matrix,  split_sets, sink);
					
					updateProgress(++mini_step);
				}
				
				/*
				* Consensus tree
				*/
				message_1 = QObject::tr(": creating consenses of all trees");
				updateProgress(message + message_1);
				split_sets.consense(num_profiles);
				updateProgress(++mini_step);
				
				/*
				* Rebuilding tree
				*/
				message_1 = QObject::tr(": rebuilding tree");
				
				updateProgress(message + message_1);
				
				Tree tree(split_sets, num_profiles, _num_bootstraps);
				
				split_sets.clear();
				
				tree.union_tree(profiles_found);
				tree.print(out, sequence_names, _num_bootstraps);
				out << std::endl;
				
				/*
				 * This is only used to display the current tree.
				 */
				std::ostringstream out_tmp;
				tree.print(out_tmp, sequence_names, _num_bootstraps);
				out_tmp << std::flush;
				
				updateProgress(++mini_step);
				
				/*
				* Search for new profiles
				*/
				message_1 = QObject::tr(": searching for new profiles");
				
				updateProgress(message + message_1);
				
				profdist::profile_set profiles;
				
				if(_pnj_method < ProfileSelectionDialog::HalfAutomaticFirstStep)
				{
					PnjGuiHandleData data;
					data.icon = _data_for_pnj->icon();
					data.alignmentName = _data_for_pnj->name();
					data.tree = out_tmp.str();
					pause(PnjGuiHandle::getGuiHandleId(), &data);
					profiles = data.profiles;
				}
				
				profiles_found.clear();
				
				size_t threshold = static_cast<size_t>(
						static_cast<float>(_threshold_minimum_bootstrap * _num_bootstraps) / 100.0);
				
				if(_threshold_minimum_bootstrap > 100)
					threshold = _num_bootstraps + 1;
				
				{
					profdist::identical_seq_set ident;
					profile.get_identical_sequences(
						ident,
						static_cast<float>(_threshold_identity) / 100.0);
					tree.find_profile(profiles_found, profiles, ident, threshold, _pnj_method);
				}
				
				updateProgress(++mini_step);
				
				/*
				* Rebuilding set of profiles
				*/
				message_1 = QObject::tr(": rebuilding set of profiles");
				
				updateProgress(message + message_1);
				
				_profile->refine(profiles_found);
				
				num_new_profiles = num_profiles - profile.get_num_profiles();
				num_profiles = profile.get_num_profiles();
				
				updateProgress(message);
				updateProgress(++mini_step);
			}
			
			out.flush();
			return true;
		}
		catch(const error& e)
		{
			GUI_MSG_N("Profile Neighbour Joining Error:" << std::endl << e);
			showErrorMessage(QObject::tr("Profile Neighbour Joining Error:\n") + e);
			return false;
		}
		catch(const std::exception& e)
		{
			GUI_MSG_N("Profile Neighbour Joining Error:" << std::endl << e.what());
			showErrorMessage(QObject::tr("Profile Neighbour Joining Error:\n") + e.what());
			return false;
		}
	}
	
}

#endif // PROFDIST_QTGUI_PROFILE_NEIGHBOUR_JOINING_OPERATION_HPP_
