#include "main_window.hpp"
#include "workspace_tree_view.hpp"
#include "workspace.hpp"
#include "feature.hpp"
#include "task.hpp"
#include "task_container_widget.hpp"
#include "operation_interface.hpp"
#include "debug.hpp"
#include "config.h"

#include "ui_about_dialog.h"

#include <cstdlib>
#include <iostream>

#include <QTreeView>
#include <QMenuBar>
#include <QMenu>
#include <QToolBar>
#include <QApplication>
#include <QHeaderView>
#include <QMessageBox>
#include <QSystemTrayIcon>
#include <QDesktopWidget>
#include <QModelIndex>
#include <QToolButton>
#include <QDockWidget>
#include <QIcon>
#include <QActionGroup>
#include <QStyleFactory>
#include <QDesktopServices>
#include <QUrl>
#include <QCloseEvent>

#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QDialog>
#include <QPixmap>
#include <QLabel>
#include <QPushButton>

namespace gui {
	
	MainWindow::MainWindow()
	: QMainWindow(0),
	_tree_view(new WorkspaceTreeView(this)),
	_workspace(new Workspace(*this)),
	_tool_button_run(new QToolButton(this)),
	_system_tray_icon(new QSystemTrayIcon(QIcon(":/images/profdist.png"))),
	_task_container_widget(new TaskContainerWidget(this))
	{
		setWindowTitle(PROFDIST_APPLICATION_NAME);
		if(mainWindow)
		{
			std::cout << "More than one MainWindow objects were created!" << std::endl;
			std::exit(1);
		}
		
		doLayout();
		mainWindow = this;
	}
	
	MainWindow::~MainWindow()
	{
		if(_system_tray_icon)
			delete _system_tray_icon;
		if(_systray_menu)
			delete _systray_menu;
		if(_tree_view)
			delete _tree_view;
		if(_task_container_widget)
			delete _task_container_widget;
		if(_workspace)
			delete _workspace;
	}
	
	QTreeView* MainWindow::workspaceTreeView()
	{
		return _tree_view;
	}
	
	void MainWindow::trackProgressOfTask(Task* task)
	{
		return _task_container_widget->trackProgressOfTask(task);
	}
	
	Workspace* MainWindow::workspace()
	{
		return _workspace;
	}
	
	void MainWindow::loadPlugins()
	{
		_workspace->loadPlugins();
	}
	
	QList<QAction*> MainWindow::defaultActions()
	{
		return QList<QAction*>() << _action_save_as << _action_remove << _action_rename << _action_view;
	}
	
	void MainWindow::doLayout()
	{
		connect(_tree_view, SIGNAL(updateFeatures()), this, SLOT(updateFeatures()));
		
		_tool_button_run->setIcon(QIcon(":/images/run.xpm"));
		_tool_button_run->setEnabled(false);
		_tool_button_run->setPopupMode(QToolButton::InstantPopup);
		
		setCentralWidget(_tree_view);
		createActions();
		createMenuBar();
		createToolBar();
		createDockWidget();
		createSystemTray();
		
		QHeaderView* header = _tree_view->header();
		header->setMovable(false);
		header->setClickable(false);
		header->setResizeMode(0, QHeaderView::ResizeToContents);
		
		int w = 500, h = 600;
		QDesktopWidget desktop;
		QRect geo = desktop.screenGeometry();
		setGeometry(geo.width()/2 - w/2, geo.height()/2 - h/2, w, h);
		
		_system_tray_icon->show();
	}
	
	void MainWindow::onHide()
	{
		hide();
	}
	
	void MainWindow::onShow()
	{
		if(!isVisible())
			show();
		setFocus();
		raise();
	}
	
	void MainWindow::onQuit()
	{	
		if(_workspace->hasRunningTasks())
		{
			switch(QMessageBox::question(this, tr("Confirm quit"), tr("There are still running tasks.\nDo you really want to quit?"), QMessageBox::Ok | QMessageBox::Cancel, QMessageBox::Cancel))
			{
				case QMessageBox::Ok: qApp->quit(); break;
				case QMessageBox::Cancel: break;
			}
		}
		else
		{
			qApp->quit();
		}
	}
	
	void MainWindow::showMessage(const QString& message, const QString& title, MessageType type)
	{
		switch(type)
		{
			case MT_ERROR: QMessageBox::critical(this, title, message); break;
			case MT_WARNING: QMessageBox::warning(this, title, message); break;
			case MT_QUESTION: QMessageBox::question(this, title, message); break;
			case MT_INFORMATION: QMessageBox::information(this, title, message); break;
		}
	}
	
	void MainWindow::ballonMessage(const QString& message)
	{
#ifndef Q_WS_MAC
		_system_tray_icon->showMessage(tr("Information"), message);
#else
		showMessage(message, tr("Information"));
#endif
	}
	
	void MainWindow::ballonMessage(const QString& title, const QString& message)
	{
#ifndef Q_WS_MAC
		_system_tray_icon->showMessage(title, message);
#else
		showMessage(message, title);
#endif
	}
	
	void MainWindow::updateFeatures()
	{
		if(_tool_button_run->menu())
		{
			delete _tool_button_run->menu();
		}
		
		std::list<Feature*> l = _workspace->getFeaturesForSelectedItem();
		
		if(l.empty())
		{
			_tool_button_run->setEnabled(false);
			return;
		}
		
		QMenu* menu = new QMenu();
		foreach(Feature* f, l)
		{
			menu->addAction(f->action());
		}
		
		_tool_button_run->setMenu(menu);
		_tool_button_run->setEnabled(true);
		
	}
	
	void MainWindow::aboutProfdist()
	{
		QDialog dia;
		Ui::AboutDialog dialog;
		dialog.setupUi(&dia);
		
		QString text = tr("<h3>") + QString(PROFDIST_APPLICATION_NAME) + " " +QString(PROFDIST_VERSION_STRING) + tr(" (Qt-Version)</h3>"
				"<p>" PROFDIST_APPLICATION_NAME " is a program to compute phylogenetic trees on<br/>"
				"DNA/RNA, DNA/RNA-Structure and Protein alignments.</p>"
				"<p>Profdist uses the following Frameworks and Libraries:"
				"<ul><li>Qt-Framework, Trolltech(<a href=\"http://www.trolltech.com\">www.trolltech.com</a>)</li>"
				"<li>Boost-C++-Library (<a href=\"http://www.boost.org\">www.boost.org</a>)</li></ul></p>"
				"<p>If you want to report bugs or improvements please contact<br/>"
				"<a href=\"mailto: ") + QString(PROFDIST_SUPPORT_EMAIL) + "\">" + QString(PROFDIST_SUPPORT_EMAIL) + "</a></p>";
		dialog.label_text->setText(text);
		connect(dialog.label_text, SIGNAL(linkActivated(const QString&)), this, SLOT(openUrl(const QString&)));
		dia.exec();
	}
	
	void MainWindow::openUrl(const QString& url)
	{
		if(!QDesktopServices::openUrl(QUrl(url)))
		{
			showMessage(tr("Could not open the specified url in a browser."));
		}
	}
	
	void MainWindow::openHelp()
	{
		openUrl(PROFDIST_ONLINE_HELP_ADDRESS);
	}
	
	void MainWindow::systemTrayActivated(QSystemTrayIcon::ActivationReason reason)
	{
		switch(reason)
		{
			case QSystemTrayIcon::Trigger:
			{
				if(isVisible() || hasFocus())
					onHide();
				else
					onShow();
			}
		}
	}
	
	void MainWindow::createActions()
	{
		_action_open = new QAction(QIcon(":/images/fileopen.png"), tr("&Open"), this);
		_action_view = new QAction(QIcon(":/images/view.png"), tr("&View"), this);
		_action_save_as = new QAction(QIcon(":/images/filesaveas.png"), tr("&Save as"), this);
		_action_remove = new QAction(QIcon(":/images/fileclose.png"), tr("&Remove"), this);
		_action_rename = new QAction(QIcon(":/images/edit.png"), tr("Re&name"), this);
		_action_quit = new QAction(QIcon(":/images/exit.png"), tr("&Quit"), this);
		_action_settings = new QAction(QIcon(":/images/package_settings.png"), tr("Settin&gs"), this);
		_action_about = new QAction(QIcon(":/images/profdist.png"), tr("About"), this);
		_action_help = new QAction(QIcon(":/images/help.png"), tr("Online Help"), this);
		_action_systray_open = new QAction(QIcon(), tr("Open Profdist"), this);
		_action_systray_hide = new QAction(QIcon(), tr("Hide Profdist"), this);
		_action_systray_quit = new QAction(QIcon(":/images/exit.png"), tr("Quit Profdist"), this);
		
		connect(_action_open, SIGNAL(triggered()), _workspace, SLOT(open()));
		connect(_action_view, SIGNAL(triggered()), _workspace, SLOT(view()));
		connect(_action_save_as, SIGNAL(triggered()), _workspace, SLOT(saveAs()));
		connect(_action_remove, SIGNAL(triggered()), _workspace, SLOT(remove()));
		connect(_action_rename, SIGNAL(triggered()), _workspace, SLOT(rename()));
		connect(_action_quit, SIGNAL(triggered()), this, SLOT(onQuit()));
		connect(_action_settings, SIGNAL(triggered()), _workspace, SLOT(settings()));
		connect(_action_about, SIGNAL(triggered()), this, SLOT(aboutProfdist()));
		connect(_action_help, SIGNAL(triggered()), this, SLOT(openHelp()));
		connect(_action_systray_open, SIGNAL(triggered()), this, SLOT(onShow()));
		connect(_action_systray_hide, SIGNAL(triggered()), this, SLOT(onHide()));
		connect(_action_systray_quit, SIGNAL(triggered()), this, SLOT(onQuit()));
	}
	
	void MainWindow::createMenuBar()
	{
		QMenu* menu = menuBar()->addMenu(tr("&File"));
		menu->addAction(_action_open);
		menu->addAction(_action_save_as);
		menu->addAction(_action_remove);
		menu->addAction(_action_rename);
		menu->addAction(_action_view);
		menu->addSeparator();
		menu->addAction(_action_quit);
		
		menu = menuBar()->addMenu(tr("&Options"));
		menu->addAction(_action_settings);
		
		menu = menuBar()->addMenu(tr("&Help"));
		menu->addAction(_action_about);
		menu->addSeparator();
		menu->addAction(_action_help);
	}
	
	void MainWindow::createToolBar()
	{
		QToolBar* toolBar = addToolBar(tr("File"));
		toolBar->addAction(_action_open);
		toolBar->addAction(_action_save_as);
		toolBar->addAction(_action_remove);
		toolBar->addAction(_action_rename);
		toolBar->addAction(_action_view);
		toolBar->addSeparator();
		toolBar->addWidget(_tool_button_run);
	}
	
	void MainWindow::createDockWidget()
	{
		QDockWidget* dockWidget = new QDockWidget(tr("Running Tasks"), this);
		dockWidget->setWidget(_task_container_widget);
		addDockWidget(Qt::BottomDockWidgetArea, dockWidget);
	}
	
	void MainWindow::createSystemTray()
	{
		_systray_menu = new QMenu();
		_systray_menu->addAction(_action_systray_open);
		_systray_menu->addAction(_action_systray_hide);
		_systray_menu->addSeparator();
		_systray_menu->addAction(_action_systray_quit);
		
		_system_tray_icon->setContextMenu(_systray_menu);
		
		connect(_system_tray_icon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)), this, SLOT(systemTrayActivated(QSystemTrayIcon::ActivationReason)));
	}
	
	void MainWindow::closeEvent(QCloseEvent* event)
	{
		// hide the main window. all task are still running. The main window
		// can be raised by clicking on the system tray icon.
		event->ignore();
		hide();
#ifndef Q_WS_MAC
		_system_tray_icon->showMessage(tr("ProfDist"), tr("ProfDist is still running in the background.\nTo restore the main window click on the icon."));
#endif
	}
	
	MainWindow* mainWindow = 0;
	
}
