#include "file_type.hpp"
#include "parser.h"
#include "parsed_alignment.hpp"
#include "main_window.hpp"
#include "alignment_data.hpp"
#include "traits.hpp"

#include "ui_trait.h"

#include <QFileInfo>

namespace gui {
	
	FileType::FileType(const QString& fileTypeName, const QString& suffixes)
	: _file_type_name(fileTypeName), _suffixes(suffixes.split(" ", QString::SkipEmptyParts))
	{}
	
	FileType::~FileType()
	{}
	
	bool FileType::containsSuffix(const QString& suffix) const
	{
		if(suffix[0] == '*' && suffix[1] == '.')
			return _suffixes.contains(QString(suffix).remove(0, 2), Qt::CaseInsensitive);
		return _suffixes.contains(suffix, Qt::CaseInsensitive);
	}
	
	QStringList FileType::suffixes() const
	{
		return _suffixes;
	}
	
	QString FileType::fileDialogFilter() const
	{
		QString filter = _file_type_name + " (";
		QStringList suffixes;
		foreach(QString suffix, _suffixes)
		{
			suffixes << ("*." + suffix);
		}
		filter += (suffixes.join(" ") + ")");
		return filter;
	}
	
	QString FileType::name() const
	{
		return _file_type_name;
	}
	
	bool FileType::operator==(const FileType& fileType) const
	{
		return fileType._file_type_name == _file_type_name;
	}
	
#define IMPLEMENT_NEW_ALIGNMENT_FILE_TYPE(class_name, suffixes, parse_method) \
	class_name::class_name() : FileType(QString(#class_name), QString(suffixes)) {} \
	DataItem* class_name::parseFile(const QString& fileName) const \
	{ \
		profdist::TraitType type; \
		if(name() != "XFasta") \
		{ \
			QDialog dia(mainWindow); \
			Ui::TraitDialog dialog; \
			dialog.setupUi(&dia); \
			dialog.FileLabel->setText(fileName); \
			dia.exec();\
			type = profdist::TraitType(dialog.FileTraits->currentIndex()); \
		} \
		else \
		{ \
			type = profdist::RnaStructure; \
		} \
		profdist::alignment a; \
		try { \
			parse_method(fileName.toStdString(), a); \
		} \
		catch(std::runtime_error& e) \
		{ \
			throw; \
		} \
		switch(type) { \
			case profdist::Rna: \
			{ \
				profdist::AlignCode<profdist::rna_traits> ac; \
				ac.read_sequences(a); \
				AlignmentData<profdist::rna_traits>* item = \
					new AlignmentData<profdist::rna_traits>(QFileInfo(fileName).baseName(), ac); \
				item->setClassificationData(a); \
				return item; \
			} \
			case profdist::RnaStructure: \
			{ \
				profdist::AlignCode<profdist::rna_structure_traits> ac; \
				ac.read_sequences(a); \
				AlignmentData<profdist::rna_structure_traits>* item = \
					new AlignmentData<profdist::rna_structure_traits>(QFileInfo(fileName).baseName(), ac); \
				item->setClassificationData(a); \
				return item; \
			} \
			case profdist::Proteine: \
			{ \
				profdist::AlignCode<profdist::protein_traits> ac; \
				ac.read_sequences(a); \
				AlignmentData<profdist::protein_traits>* item = \
					new AlignmentData<profdist::protein_traits>(QFileInfo(fileName).baseName(), ac); \
				item->setClassificationData(a); \
				return item; \
			} \
		} \
	}
	
	IMPLEMENT_NEW_ALIGNMENT_FILE_TYPE(Fasta, "fas fasta", parse_fasta)
	IMPLEMENT_NEW_ALIGNMENT_FILE_TYPE(Embl, "embl", parse_embl)
	IMPLEMENT_NEW_ALIGNMENT_FILE_TYPE(XFasta, "xfasta bdb", parse_bdb);
	
#undef IMPLEMENT_NEW_ALIGNMENT_FILE_TYPE
	
}

