#include "distance_data.hpp"
#include "debug.hpp"
#include <stdexcept>

namespace gui {
	
	DistanceData::DistanceData(
		const QString& name,
		const std::vector<std::string>& sequenceNames,
  		const std::vector<profdist::distance_matrix>& distanceMatrices)
	: _sequence_names(sequenceNames), _distance_matrices(distanceMatrices), _max_name_len(0)
	{
		this->setName(name);
		// search for the longest name and store it in _max_name_length
		for(sequence_names_t::const_iterator i = _sequence_names.begin(), e = _sequence_names.end(); i != e; i++)
		{
			if((*i).size() > _max_name_len)
			{
				_max_name_len = (*i).size();
			}
		}
	}
	
	DistanceData::~DistanceData()
	{
		GUI_MSG_N("DistanceData object destroyed");
	}
	
	bool DistanceData::writeData(std::ostream& out, profdist::FileType /*fileType*/)
	{
		distance_matrices_t::const_iterator i = _distance_matrices.begin();
		size_t nRows = i->nRows(), nCols = i->nCols();
		
		for(distance_matrices_t::const_iterator e = _distance_matrices.end();
			i != e; i++)
		{
			out << nRows << " " << nCols << std::endl;
			size_t n_rows = i->nRows();
			sequence_names_t::const_iterator names_iter = _sequence_names.begin();
			sequence_names_t::const_iterator names_end = _sequence_names.end();
			for(int row = 0; row < nRows && names_iter != names_end; row++, names_iter++)
			{
				out << setw(_max_name_len) << *names_iter << "  " << flush;
				for(int col = 0; col < nCols; col++)
				{
					out << (*i)(row, col);
					if(col < nCols - 1)
						out << " ";
				}
				out << std::endl;
			}
			out << std::endl << std::endl;
		}
		return true;
	}
	
	unsigned int DistanceData::type() const
	{
		return DataType::DistanceMatrix;
	}
	
	QIcon DistanceData::icon() const
	{
		return _distance_matrices.size() > 1 ? QIcon(":/images/64x64/distances.png") : QIcon(":/images/64x64/distance.png");
	}
	
	QString DistanceData::info() const
	{
		return QString().setNum(_distance_matrices.size()) + QObject::tr(" Distance Matrices");
	}
	
	QString DistanceData::dataToString() const
	{
		QString out;
		distance_matrices_t::const_iterator i = _distance_matrices.begin();
		for(distance_matrices_t::const_iterator e = _distance_matrices.end(); i != e; i++)
		{
			size_t n_rows = i->nRows();
			for(int row = 0; row < n_rows; row++)
			{
				size_t n_cols = i->nCols();
				for(int col = 0; col < n_cols; col++)
				{
					out += QString().setNum( (*i)(row, col) );
					if(col < n_cols - 1)
						out += " ";
				}
				out += "\n";
			}
			out += "\n\n";
		}
		return out;
	}
	
	DataForNj::sequence_names_t DistanceData::getSequenceNames() const
	{
		return _sequence_names;
	}
	
	profdist::distance_matrix DistanceData::getDistanceMatrix(size_t index) const
	{
		if(index >= _distance_matrices.size())
			throw std::runtime_error("no valid index, index too big");
		
		return _distance_matrices[index];
	}
	
	size_t DistanceData::getNumMatrices() const
	{
		return _distance_matrices.size();
	}
	
}
