#ifndef PROFDIST_QTGUI_BOOTSTRAP_OPERATION_HPP_
#define PROFDIST_QTGUI_BOOTSTRAP_OPERATION_HPP_

#include <iostream>
#include <stdexcept>
#include "operation_interface.hpp"
#include "operation_accessors.hpp"
#include "bootstrap_data.hpp"
#include "task.hpp"
#include "data_type.hpp"
#include "debug.hpp"
#include "bootstrap.h"
#include "main_window.hpp"

#include "ui_bootstrap_dialog.h"

#include <QPixmap>

namespace gui {
	
	template<typename Traits>
	class BootstrapOperation : public OperationInterface {
		public:
									BootstrapOperation();
									~BootstrapOperation();
			void					execute();
			bool					setDataItem(DataItem* item);
			OperationInterface*		clone();
			bool					preExecution();
			
		private:
			size_t						_num_bootstraps;
			DataForBootstrap<Traits>	*_data_for_bootstrap;
	};
	
	template<typename Traits>
	BootstrapOperation<Traits>::BootstrapOperation()
	: _num_bootstraps(0), _data_for_bootstrap(0)
	{}
	
	template<typename Traits>
	BootstrapOperation<Traits>::~BootstrapOperation()
	{
		GUI_MSG_OBJ_DESTROYED(BootstrapOperation);
	}
	
	template<typename Traits>
	void BootstrapOperation<Traits>::execute()
	{
		if(!_data_for_bootstrap)
		{
			QString errorMessage = QObject::tr(
					"error during bootstrap: no alignment data available. "
					"Has the data been set with setDataItem()?");
			GUI_MSG_N(qPrintable(errorMessage));
			if(mainWindow)
				mainWindow->showMessage(errorMessage);
			return;
		}
		
		QString message = QObject::tr("Bootstrapping ") +
				_data_for_bootstrap->name() + QObject::tr(" [") +
				QString().setNum(_num_bootstraps) + QObject::tr("]");
		
		setMaximumValue(_num_bootstraps);
		updateProgress(0);
		updateProgress(message);
		
		const profdist::AlignCode<Traits> a = _data_for_bootstrap->alignCode();
			
		BootstrapData<Traits>* bootstrapItem =
				new BootstrapData<Traits>(_data_for_bootstrap->name() + "_bootstrap", _num_bootstraps);
		
		bootstrapItem->openStream();
		
		try {
			for(int i = 0; i < _num_bootstraps && !isCanceled(); i++)
			{
				profdist::AlignCode<Traits> b;
				profdist::bootstrap(a, b);
				bootstrapItem->addBootstrapAlignment(b);
				updateProgress(i);
			}
		}
		catch(std::runtime_error& e)
		{
			GUI_MSG_N("error during bootstrap: " << e.what());
			if(mainWindow)
			{
				mainWindow->showMessage(QObject::tr("Error during bootstrap: ") + QString(e.what()));
			}
			if(bootstrapItem)
			{
				delete bootstrapItem;
			}
		}
		
		bootstrapItem->closeStream();
		
		if(isCanceled())
		{
			if(bootstrapItem)
			{
				delete bootstrapItem;
			}
		}
		else
		{
			addDataItem(bootstrapItem, _data_for_bootstrap);
		}
		
		_data_for_bootstrap->decrementUseCount();
	}
	
	template<typename Traits>
	bool BootstrapOperation<Traits>::setDataItem(DataItem* item)
	{
		if(!item)
			return false;
		
		_data_for_bootstrap = dynamic_cast<DataForBootstrap<Traits>* >(item);
		if(!_data_for_bootstrap)
		{
			return false;
		}
		_data_for_bootstrap->incrementUseCount();
		return true;
	}
	
	template<typename Traits>
	OperationInterface* BootstrapOperation<Traits>::clone()
	{
		BootstrapOperation<Traits>* op = new BootstrapOperation<Traits>();
		
		op->_num_bootstraps = _num_bootstraps;
		op->_data_for_bootstrap = _data_for_bootstrap;
		op->setMaximumValue(maximumValue());
		
		return op;
	}
	
	template<typename Traits>
	bool BootstrapOperation<Traits>::preExecution()
	{
		QDialog dia(mainWindow);
		Ui::BootstrapDialog dialog;
		dialog.setupUi(&dia);
		dialog.labelAlignmentIcon->setPixmap(
				 _data_for_bootstrap->icon().pixmap(
				dialog.labelAlignmentIcon->width(),
				dialog.labelAlignmentIcon->height()));
		dialog.labelAlignmentName->setText(_data_for_bootstrap->name());
		if(QDialog::Rejected == dia.exec())
		{
			return false;
		}
		
		_num_bootstraps = dialog.spinBoxNumBootstraps->value();
		
		return true;
	}
	
}

#endif // PROFDIST_QTGUI_BOOTSTRAP_OPERATION_HPP_

