#ifndef PROFDIST_QTGUI_BOOTSTRAP_DATA_HPP_
#define PROFDIST_QTGUI_BOOTSTRAP_DATA_HPP_

#include <stdexcept>
#include <vector>
#include <cstdlib>
#include <fstream>
#include "operation_accessors.hpp"
#include "spirit_helper.h"
#include "types.h"
#include "debug.hpp"
#include "helper.hpp"

#include <boost/cstdlib.hpp>
#include <boost/spirit/phoenix.hpp>
#include <boost/spirit/core.hpp>
#include <boost/spirit/core/primitives/primitives.hpp>
#include <boost/spirit/utility/grammar_def.hpp>
#include <boost/spirit/utility/lists.hpp>
#include <boost/spirit/utility/chset.hpp>
#include <boost/spirit/iterator/file_iterator.hpp>
#include <boost/spirit/iterator/position_iterator.hpp>
#include <boost/spirit/core/composite/epsilon.hpp>
#include <boost/spirit/actor/push_back_actor.hpp>
#include <boost/spirit/actor/increment_actor.hpp>
#include <boost/spirit/actor/clear_actor.hpp>

#include <QTemporaryFile>

namespace gui {
	
	template<typename Traits>
	class BootstrapData : public DataForDistance<Traits> {
		public:
			typedef typename DataForDistance<Traits>::index_type index_type;
			
			BootstrapData(const QString& name, size_t numBootstraps);
			~BootstrapData();
			QIcon icon() const;
			QString info() const;
			QString dataToString() const;
			size_t numBootstraps() const;
			profdist::AlignCode<Traits> bootstrapData(index_type index);
			bool openStream();
			void closeStream();
			void addBootstrapAlignment(profdist::AlignCode<Traits>& align);
			unsigned int type() const;
			bool writeData(std::ostream& out, profdist::FileType type);
			
		private:
			QTemporaryFile					_tmp_file;
			std::ofstream					_out;
			size_t							_act_index;
			std::vector<index_type>			_positions;
	};
	
	template<typename Traits>
	BootstrapData<Traits>::BootstrapData(const QString& name, size_t numBootstraps)
	: _tmp_file("profdist_bootstrap_XXXXXX.txt"), _out(), _act_index(0),  _positions(numBootstraps)
	{
		this->setName(name);
		_tmp_file.open();
	}
	
	template<typename Traits>
	BootstrapData<Traits>::~BootstrapData()
	{
		GUI_MSG_N("BootstrapData object destroyed");
	}
	
#ifdef Q_OS_WIN
	template<> QIcon BootstrapData<profdist::rna_traits>::icon() const;
	template<> QIcon BootstrapData<profdist::rna_structure_traits>::icon() const;
	template<> QIcon BootstrapData<profdist::protein_traits>::icon() const;
#endif
	
	template<typename Traits>
	QString BootstrapData<Traits>::dataToString() const
	{
		QString data;
		BootstrapData<Traits>* d = const_cast<BootstrapData<Traits>*>(this);
		for(int i = 0; i < _positions.size(); i++)
		{
			data.append(alignmentToQString(d->bootstrapData(i)));
		}
		return data;
	}
	
	template<typename Traits>
	size_t BootstrapData<Traits>::numBootstraps() const
	{
		return _positions.size();
	}
	
	template<typename Traits>
	profdist::AlignCode<Traits> BootstrapData<Traits>::bootstrapData(index_type index)
	{
		std::ifstream in(_tmp_file.fileName().toStdString().c_str(), std::ios_base::binary);
		
		if(!in)
			throw std::runtime_error("Error while opening from bootstrap file '"
				+ std::string(_tmp_file.fileName().toStdString()) + "': could not open file");
		
		if(index >= _positions.size())
			throw std::runtime_error("Error while reading from bootstrap file '"
				+ std::string(_tmp_file.fileName().toStdString()) + "': requested index greater than number of bootstraps");
		
		index_type start_pos = 0;
		index_type end_pos = 0;
		
		start_pos = _positions[index];
		
		if(index < _positions.size() - 1)
			end_pos = _positions[index + 1];
		else
		{
			in.seekg(0, std::ios_base::end);
			end_pos = in.tellg();
		}
		
		if(!in.seekg(_positions[index]))
			throw std::runtime_error("Error while reading from bootstrap file '"
				+ std::string(_tmp_file.fileName().toStdString()) + "': could not seek to position");
		
		index_type data_size = end_pos - start_pos;
		char * data = new char[data_size +1];
		data[data_size] = 0;
		
		if(!in.read(data, data_size))
			throw std::runtime_error("Error while reading from bootstrap file '"
				+ std::string(_tmp_file.fileName().toStdString()) + "': could not read amount of data");
		
		in.close();
		
		profdist::alignment align;
		
		try
		{
			parseAlignmentFromString(data, &data[data_size], align, _tmp_file.fileName().toStdString(), Traits());
		}
		catch(const std::runtime_error& e)
		{
			throw;
		}
		
		profdist::AlignCode<Traits> alignCode;
		alignCode.read_sequences(align);
		
		if(data)
			delete [] data;
		
		return alignCode;
	}
	
	template<typename Traits>
	bool BootstrapData<Traits>::openStream()
	{
		_out.open(_tmp_file.fileName().toStdString().c_str());
	}
	
	template<typename Traits>
	void BootstrapData<Traits>::closeStream()
	{
		_out.close();
	}
	
	template<typename Traits>
	void BootstrapData<Traits>::addBootstrapAlignment(profdist::AlignCode<Traits>& align)
	{
		if(_act_index >= _positions.size())
			throw std::runtime_error("addBootstrapAlignment: could not add further bootstrap data sets.");
		
		_out.seekp(0, std::ios_base::end);
		std::ofstream::pos_type s = _out.tellp();
		
		profdist::write_file(align, _out, profdist::Fasta);
		
		_positions[_act_index++] = s;
	}
	
	template<typename Traits>
	bool BootstrapData<Traits>::writeData(std::ostream& out, profdist::FileType type)
	{
		bool b;
		
		for(int i = 0; i < _positions.size(); i++)
			b = writeAlignmentToFile(out, bootstrapData(i));
		
		return b;
	}
	
}

#endif // PROFDIST_QTGUI_BOOTSTRAP_DATA_HPP_

