/***************************************************************************
 *   Copyright (C) 2005 by Andreas Pokorny                                 *
 *   andreas.pokorny@biozentrum.uni-wuerzburg.de                           *
 *                                                                         *
 *   This file is part of profdist and cbcanalyzer                         *
 *                                                                         *
 *   Both profdist and cbcanalyzer are free software; you can redistribute *
 *   it and/or modify it under the terms of the GNU General Public License *
 *   as published by the Free Software Foundation; either version 2 of the *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   Profdist and cbcanalyzer are distributed in the hope that it will be  *
 *   useful, but WITHOUT ANY WARRANTY; without even the implied warranty   *
 *   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <algorithm>
#include <iterator>
#include "profile.h"
#include "dnamapper.h"

profdist::Profile::Profile( std::size_t sites, std::size_t prof, std::size_t rows) 
  : num_sites( sites ), num_profiles( prof ), num_rows( rows ), profile_array(num_profiles * num_sites* num_rows, 0.0 )
{
}




void profdist::Profile::init( std::size_t sites, std::size_t prof, std::size_t rows )
{
  num_sites = sites; 
  num_profiles = prof;
  num_rows = rows;
  profile_array.resize( num_profiles*num_sites*num_rows);
  std::memset( &(profile_array[0]), 0, sizeof(double)*profile_array.size() );
}

void profdist::Profile::refine( tree_types::profile_map const& list )
{
  Profile empty( get_num_sites(), list.size(), num_rows ); // creating empty profile 
  empty.swap(*this); // swap to keep current and make "this" empty

  Profile const& backup( empty );
  
  for( tree_types::profile_map::const_iterator it = list.begin(), e = list.end();  it != e; ++it  )
  {
    SingleProfile dest(get_profile( it->first ));
    CountedSets::set_type const& splits = it->second->get_split_set();
    
    if( splits.size() )
    {
      for( Node::set_type::const_iterator set_it = splits.begin(), set_e = splits.end(); set_it != set_e;  ++set_it ) 
        dest += backup.get_profile( *set_it  );
//      dest.normalize( splits.size() );
      dest.normalize();
    }
    else // single node => there was no change in that step => just copy the data
    {
      ConstSingleProfile source( backup.get_profile( it->second->get_reference_position() ) );
      dest = source;
    }
  }
}
void profdist::Profile::SingleProfile::normalize()
{
  for( std::size_t i = 0; i != num_sites; ++i )  {
    double val = 0.0;
    std::size_t off_set = i * num_rows;
    for( std::size_t j = 0; j < num_rows; ++j )
      val += data[ off_set + j ];
    if( val > 0.00001 )
      for( std::size_t j = 0; j < num_rows; ++j )
        data[ off_set + j ] /= val;
  }
}
#if 0

void profdist::Profile::SingleProfile::normalize( size_t num_entries )
{
  for( std::size_t i = 0; i != num_sites; ++i )  {
    double val = 0.0;
    std::size_t off_set = i * num_rows;
    for( std::size_t j = 0; j < num_rows; ++j )
      val += data[ off_set + j ];
    if( val > 0.00001 )
      for( std::size_t j = 0; j < num_rows; ++j )
        data[ off_set + j ] /= val;
  }

/*  for( std::size_t i = 0; i != num_sites*num_rows; ++i ) 
    data[i] /= value_type(num_entries);*/
}

#endif
profdist::Profile::SingleProfile::SingleProfile( value_type* d, std::size_t rows, std::size_t sites )
  : data( d ), num_rows( rows ), num_sites( sites ) { }


profdist::Profile::SingleProfile profdist::Profile::get_profile(std::size_t profile_index ) 
{
  return SingleProfile( &*(profile_array.begin() + profile_index * num_sites * num_rows), num_rows, num_sites);
}

profdist::Profile::SingleProfile profdist::Profile::operator[]( std::size_t profile_index ) 
{
  return SingleProfile( &*(profile_array.begin() + profile_index * num_sites * num_rows), num_rows, num_sites );
}

profdist::Profile::ConstSingleProfile::ConstSingleProfile( profdist::Profile::SingleProfile const& ref )
  : data( ref.data ), num_rows( ref.num_rows ), num_sites( ref.num_sites ) { }
profdist::Profile::ConstSingleProfile::ConstSingleProfile( value_type const* d, std::size_t rows, std::size_t sites )
  : data( d ), num_rows( rows ), num_sites( sites ) { }

const size_t profdist::Profile::ConstSingleProfile::get_num_sites() const{
  return num_sites;
}
const size_t profdist::Profile::ConstSingleProfile::get_num_rows() const{
  return num_rows;
}

profdist::Profile::ConstSingleProfile profdist::Profile::get_profile(std::size_t profile_index )  const
{
  return ConstSingleProfile( &*(profile_array.begin() + profile_index * num_sites * num_rows), num_rows, num_sites);
}

profdist::Profile::ConstSingleProfile profdist::Profile::operator[]( std::size_t profile_index ) const
{
  return ConstSingleProfile( &*(profile_array.begin() + profile_index * num_sites * num_rows), num_rows, num_sites );
}

profdist::Profile::SingleProfile& profdist::Profile::SingleProfile::operator=( ConstSingleProfile const& right )
{
  std::memcpy( data, right.data, num_sites*sizeof(profdist::Profile::value_type)*num_rows );
  return *this;
}
profdist::Profile::SingleProfile& profdist::Profile::SingleProfile::operator+=( ConstSingleProfile const& right )
{
  std::size_t i = num_rows*num_sites;
  do {
    --i;
    data[i] += right.data[i];
  }
  while( i );
  return *this;
}

const std::size_t profdist::Profile::get_num_sites() const
{
  return num_sites;
}
const std::size_t profdist::Profile::get_num_profiles() const
{
  return num_profiles;
}

const std::size_t profdist::Profile::get_num_rows() const
{
  return num_rows;
}

/*std::ostream& operator << ( std::ostream & out, Profile const& obj )
{
  // WARNING: This code depends on the Profile memory layout
  out << obj.get_num_sequences() << ' ' << obj.get_num_sites() << '\n';
  
  profdist::Profile::const_iterator b = obj.begin();
  for( profdist::Profile::const_iterator it = obj.begin(), e = obj.end(); 
      it != e; ++it )
  {
    out << setw(5) << *it;
    if( ( it - b  ) % obj.get_num_rows() == 0 ) 
      out << '\n';
  }
  out << std::endl;
  return out;
}
*/

void profdist::Profile::swap( Profile& rhs )
{
  std::swap(num_sites,rhs.num_sites);
  std::swap(num_profiles,rhs.num_profiles);
  std::swap(num_rows,rhs.num_rows );
  std::swap(profile_array,rhs.profile_array);
}

void profdist::swap( Profile &lhs, Profile &rhs )
{
  lhs.swap(rhs);
}

std::ostream & profdist::operator<<( std::ostream & o, Profile const& obj )
{
  for( std::size_t i = 0; i < obj.get_num_profiles(); ++i ) {
    o << i << ".\n";
    profdist::Profile::ConstSingleProfile prof = obj.get_profile( i );
    for( std::size_t j = 0; j < obj.get_num_rows(); ++j ) {
      for( std::size_t k = 0; k < obj.get_num_sites(); ++k ) 
        o << prof.get_value(k,j)<< '|';
      o << std::endl;
    }
  }
  return o;

}

std::ostream & operator<<( std::ostream & o, profdist::Profile::ConstSingleProfile const& prof )
{
  for( std::size_t j = 0; j < prof.get_num_rows(); ++j ) {
    for( std::size_t k = 0; k < prof.get_num_sites(); ++k ) 
      o << prof.get_value(k,j)<< '|';
    o << std::endl;
  }
  return o;

}

void profdist::Profile::get_identical_sequences( profdist::identical_seq_set & ids, float percentual_identity  ) const {
  for( std::size_t i = 0, e = num_profiles; i != e - 1; ++i )  {
    for( std::size_t j = i + 1; j != e; ++j ) {
      float id = 0;
      profdist::Profile::ConstSingleProfile prof_i = get_profile( i )
        , prof_j = get_profile( j );
      for( std::size_t element_index = 0; element_index < num_sites; ++element_index )
        for( std::size_t row_index = 0; row_index < num_rows; ++row_index ) {
          id += abs( prof_j.get_value( element_index, row_index ) - prof_i.get_value( element_index, row_index ) );
        }
      if( percentual_identity <  ( float(num_sites) - (id*0.5f) ) / float(num_sites) ) {
        ids.insert( std::make_pair( i, j ) );
        ids.insert( std::make_pair( j, i ) );
      }
    }
  }
}


