/***************************************************************************
 *   Copyright (C) 2005 by Andreas Pokorny                                 *
 *   andreas.pokorny@biozentrum.uni-wuerzburg.de                           *
 *                                                                         *
 *   This file is part of profdist and cbcanalyzer                         *
 *                                                                         *
 *   Both profdist and cbcanalyzer are free software; you can redistribute * 
 *   it and/or modify it under the terms of the GNU General Public License * 
 *   as published by the Free Software Foundation; either version 2 of the * 
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   Profdist and cbcanalyzer are distributed in the hope that it will be  *
 *   useful, but WITHOUT ANY WARRANTY; without even the implied warranty   *
 *   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PROFDIST_NEWICK_H_INCLUDED
#define PROFDIST_NEWICK_H_INCLUDED

// #define BOOST_SPIRIT_DEBUG
#include <map>
#include <boost/spirit/core.hpp>
#include <boost/spirit/symbols.hpp>
#include <boost/spirit/attribute.hpp>
#include <boost/spirit/dynamic.hpp>
#include <boost/spirit/utility/chset.hpp>
#include <boost/spirit/utility/lists.hpp>
#include <boost/spirit/iterator/file_iterator.hpp>
#include <boost/spirit/utility/grammar_def.hpp>
#include <boost/spirit/iterator/position_iterator.hpp>
#include <boost/lambda/bind.hpp>
#include "parsed_tree.h"

/**
 * Parses newick 
 */
namespace profdist{
  
struct newick_grammar
:   public boost::spirit::grammar<newick_grammar>
{
  typedef std::map<std::string,size_t> sequence_lookup;
  parsed_node::node_ptr root_node;
  boost::shared_ptr<sequence_lookup> seqs;
  newick_grammar( parsed_node::node_ptr root, boost::shared_ptr<sequence_lookup> s ) : root_node(root), seqs(s) {}
  
  /**
   * grammar definition
   */
  template<typename ScannerT>
  struct definition
  {
    typedef boost::spirit::rule<ScannerT> rule_t;
    parsed_node::node_ptr current_node;
    std::list<parsed_node::node_ptr> stack;

    rule_t newick, node;

    void add_node( newick_grammar const& self) // , typename ScannerT::iterator_t begin, typename ScannerT::iterator_t end ) 
    {
      current_node->childs.push_back( parsed_node::node_ptr(new parsed_node) );
      stack.push_back( current_node );
      current_node = current_node->childs.back();
    }
    void leave_level() {
      current_node = stack.back();
      stack.pop_back();
    }

    void set_extra_data(  typename ScannerT::iterator_t begin, typename ScannerT::iterator_t end ) {
      current_node->extra_data.assign(begin, end);
    }

    void set_leaf( newick_grammar const& self, typename ScannerT::iterator_t begin, typename ScannerT::iterator_t end ) {
      std::string name( begin, end );
      sequence_lookup::const_iterator it = self.seqs->find( name );
      if( it == self.seqs->end() ) {
        current_node->sequence_id = self.seqs->size();
        self.seqs->insert( std::pair<std::string,std::size_t>( name, self.seqs->size() ) );
      }
      else 
        current_node->sequence_id = it->second;
    }

    definition(newick_grammar const& self)
      : current_node(self.root_node)
    {
      using boost::spirit::assign_a;
      using boost::spirit::ch_p;
      using boost::spirit::chset;
      using boost::spirit::blank_p;
      using boost::spirit::chset;
      using boost::spirit::eps_p;
      using boost::spirit::alpha_p;
      using boost::lambda::bind;
      using boost::lambda::_1;
      using boost::lambda::_2;


      node = 
        ( 
         ch_p('(')
         [bind(&newick_grammar::definition<ScannerT>::add_node, this, self  )] 
         >> *list_p( 
           node
           , ch_p(',') 
           [bind(&newick_grammar::definition<ScannerT>::add_node, this, self )] 
           )
         >> ch_p(')')
         |
         (+~chset<>(":,)("))
         [ bind( &newick_grammar::definition<ScannerT>::set_leaf, this, self, _1, _2) ]
        )
        >> (
            !( 
              (+~chset<>(",)(") )
              [ bind( &newick_grammar::definition<ScannerT>::set_extra_data, this, _1, _2) ]
             )
           )
        [bind(&newick_grammar::definition<ScannerT>::leave_level, this )] 
        ;
      newick =
         ch_p('(')
         [bind(&newick_grammar::definition<ScannerT>::add_node, this, self  )] 
         >> *list_p( 
           node
           , ch_p(',') 
           [bind(&newick_grammar::definition<ScannerT>::add_node, this, self )] 
           )
        >> ");"
        ;

#ifdef BOOST_SPIRIT_DEBUG
      BOOST_SPIRIT_DEBUG_RULE( newick );
      BOOST_SPIRIT_DEBUG_RULE( node );
#endif
    }

    rule_t const& start() const
    {
      return newick;
    }
  };
};

}

#endif

