#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdexcept>

#if !defined(PROFDIST_FILEITERATOR_STD)
#  if (defined(WIN32) || defined(_WIN32) || defined(__WIN32__))
#    define PROFDIST_FILEITERATOR_WINDOWS
#  elif defined(HAVE_UNISTD_H)
extern "C"
{
#    include <unistd.h>
}
#    ifdef _POSIX_MAPPED_FILES
#      define PROFDIST_FILEITERATOR_POSIX
#    endif // _POSIX_MAPPED_FILES
#  endif // HAVE_UNISTD_H
#  if !defined(PROFDIST_FILEITERATOR_WINDOWS) && \
      !defined(PROFDIST_FILEITERATOR_POSIX)
#    define PROFDIST_FILEITERATOR_STD
#  endif
#endif // PROFDIST_FILEITERATOR_STD
#ifdef PROFDIST_FILEITERATOR_WINDOWS
#  define WIN32_LEAN_AND_MEAN
#  include <windows.h>
#endif

#include <cstdio>
#include <boost/shared_ptr.hpp>

#ifdef PROFDIST_FILEITERATOR_WINDOWS
#  include <boost/type_traits/remove_pointer.hpp>
#endif

#ifdef PROFDIST_FILEITERATOR_POSIX
#  include <sys/types.h> // open, stat, mmap, munmap
#  include <sys/stat.h>  // stat
#  include <fcntl.h>     // open
#  include <unistd.h>    // stat, mmap, munmap
#  include <sys/mman.h>  // mmap, mmunmap
#endif

#ifdef PROFDIST_FILEITERATOR_STD
#include <fstream>
#endif

#include "file_resources.hpp"

using namespace std;

profdist::memory_resource::memory_resource( std::string const& filename ) 
  : mem(0), length(0) 
{
#ifdef PROFDIST_FILEITERATOR_STD
  ifstream is(filename.c_str(), ios::binary );
  if( ! is ) 
    throw runtime_error( "Could not open file." );
  // get length of file:
  is.seekg (0, ios::end);
  length = is.tellg();
  is.seekg (0, ios::beg);

  // allocate memory:
  mem = new char[length];

  // read data as a block:
  if(!  is.read( mem,length ) ) {
    is.close();
    delete []mem;
    throw runtime_error("Failure reading file.");
  }

  is.close();

#elif defined(PROFDIST_FILEITERATOR_POSIX)
  // open the file
  int fd = open(filename.c_str(),
#ifdef O_NOCTTY
      O_NOCTTY | // if stdin was closed then opening a file
      // would cause the file to become the controlling
      // terminal if the filename refers to a tty. Setting
      // O_NOCTTY inhibits this.
#endif
      O_RDONLY);

  if( fd == -1 )
    throw runtime_error("Could not open file.");

  // call fstat to find get information about the file just
  // opened (size and file type)
  struct stat stat_buf;
  if( (fstat(fd, &stat_buf) != 0) || !S_ISREG(stat_buf.st_mode) )
  {   // if fstat returns an error or if the file isn't a
    // regular file we give up.
    close(fd);
    throw runtime_error("Could not stat file as a regular file.");
  }

  // perform the actual mapping
  mem = static_cast<char*>(mmap(0, stat_buf.st_size, PROT_READ, MAP_SHARED, fd, 0));
  length = stat_buf.st_size;
  // it is safe to close() here. POSIX requires that the OS keeps a
  // second handle to the file while the file is mmapped.
  close(fd);

  if( mem == MAP_FAILED)
    throw runtime_error("File mapping failed.");

#else
  HANDLE file_handle = ::CreateFileA(
      filename.c_str(),
      GENERIC_READ,
      FILE_SHARE_READ,
      NULL,
      OPEN_EXISTING,
      FILE_FLAG_SEQUENTIAL_SCAN,
      NULL
      );

  if (file_handle == INVALID_HANDLE_VALUE)
    throw runtime_error("Failed to open file");

  // Store the size of the file, it's used to construct
  //  the end iterator
  length = ::GetFileSize(file_handle, NULL);

  HANDLE map_handle = ::CreateFileMapping(
      file_handle,
      NULL,
      PAGE_READONLY,
      0, 0,
      NULL
      );

  if (map_handle == NULL)
  {
    ::CloseHandle(file_handle);
    throw runtime_error("Failed to open file mapping");
  }

  LPVOID memory = ::MapViewOfFile(
      map_handle,
      FILE_MAP_READ,
      0, 0, 0
      );

  if (memory == NULL)
  {
    ::CloseHandle(map_handle);
    ::CloseHandle(file_handle);
    throw runtime_error("Failed to open view of file");
  }

  // We hold both the file handle and the memory pointer.
  // We can close the map_handle handle now because Windows holds internally
  //  a reference to it since there is a view mapped.
  ::CloseHandle(map_handle);

  // It seems like we can close the file handle as well (because
  //  a reference is hold by the filemap object).
  ::CloseHandle(file_handle);

  // Store the handles inside the shared_ptr (with the custom destructors)
  mem = static_cast<char *>(memory);
#endif
}

profdist::memory_resource::~memory_resource() {
#ifdef PROFDIST_FILEITERATOR_STD
  delete [] mem;
#elif defined(PROFDIST_FILEITERATOR_POSIX)
  munmap( mem, length );
#else
  UnmapViewOfFile( mem );
#endif
}
