/***************************************************************************
 *   Copyright (C) 2005 by Andreas Pokorny                                 *
 *   andreas.pokorny@biozentrum.uni-wuerzburg.de                           *
 *                                                                         *
 *   This file is part of profdist and cbcanalyzer                         *
 *                                                                         *
 *   Both profdist and cbcanalyzer are free software  you can redistribute *
 *   it and/or modify it under the terms of the GNU General Public License *
 *   as published by the Free Software Foundation  either version 2 of the *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   Profdist and cbcanalyzer are distributed in the hope that it will be  *
 *   useful, but WITHOUT ANY WARRANTY  without even the implied warranty   *
 *   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program  if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <fstream>
#include <iostream>
#include <stdexcept>
#include <vector>
#include <string>
#include <boost/program_options.hpp>
#include <boost/algorithm/string/trim.hpp>
#include "types.h"
#include "distance.h"
#include "aligncode.h"
#include "traits.hpp"
using namespace std;
using namespace profdist;
int main( int argc, char **argv )
{
  try {
    namespace po = boost::program_options;
    po::options_description desc("Usage");
    desc.add_options()
      ("help", "produce help message")
      ("output,o", po::value<string>(), "set output file name, otherwise print to stdout")
      ("tsv",  "print matrix as tab separated values")
      ("csv",  "print matrix as comma separated values")
      ("N-matrix,n",  po::value<string>(), "base file name for n-matrices of all sequences compared, otherwise no output of these will be generated")
      ("rate-matrix,q", po::value<string>(), "Q or rate matrix to be used")
      ("correction,c", po::value<string>(), "values: jukes,gtr,uncorp,kimura,logdet\njukes is default")
      ;
    po::options_description hidden("Hidden options");
    hidden.add_options()
          ("input-file", po::value<string>(), "input file")
              ;        
    po::options_description all_opts;
    all_opts.add(desc).add(hidden);

    po::positional_options_description pd; 
    pd.add("input-file", 1);

    po::variables_map vm;
    po::store(po::command_line_parser(argc, argv).options(all_opts).positional(pd).run(), vm);
    po::notify(vm);    

    if( vm.count("help") ){
      cout << desc << "\n";
      return 0;
    }

    ofstream out;
    if( vm.count("output") )
    {
      out.open(vm["output"].as<string>().c_str());
      if( ! out ) 
        throw runtime_error("Error opening output file");
    }

    CorrectionModel model = Jukes;
    if( vm.count("correction") ) {
      std::cout << vm["correction"].as<string>();
      std::string correction =  vm["correction"].as<string>();
      if( correction == "jukes" )       {model = profdist::Jukes; }
      else if( correction == "kimura")  { model = profdist::Kimura;}
      else if( correction == "logdet")  { model = profdist::LogDet; }
      else if( correction == "uncorp")  { model = profdist::UncorrectedP; }
      else if( correction == "gtr")     { model = profdist::GTR; }
      else { throw logic_error("bad distance model"); }
    }

    protein_traits::rate_matrix q(0);
    if( vm.count("rate-matrix") ) {
      ifstream in( vm["rate-matrix"].as<string>().c_str() );
      if( !in || read_rate_matrix<protein_traits>( in, q ) )
        throw runtime_error("illegal rate matrix format or file not found.");
    }

    
    if( vm.count("input-file") )
    {
      profdist::alignment align;
      parse_fasta( vm["input-file"].as<string>(), align );
      AlignCode<protein_traits> a;
      a.read_sequences( align );
      distance_matrix matrix;
      compute_distance( a, matrix, q , model, Taylor, Robust );
      vector<string> names = a.get_sequence_names();

      if( vm.count("tsv") )
        profdist::print_distance_matrix_tsv( 
            vm.count("output") ? out : cout  
            , matrix
            , names.begin()
            , names.end(), 
            '\t', 
            true 
            );
      else if( vm.count("csv") )
        profdist::print_distance_matrix_tsv(
            vm.count("output") ? out : cout 
            , matrix
            , names.begin()
            , names.end()
            , ','
            , true 
            );

      else 
        profdist::print_distance_matrix( 
            vm.count("output") 
            ? out
            : cout 
            , matrix
            , names.begin()
            , names.end()
            , true 
            );

    }
    return 0;

    cout << desc << "\n";
  }
  catch( runtime_error const& e )  {
    cerr << e.what() << endl;
  }
  catch(  exception const & e ) {
    cerr << e.what() << endl;
  } 
  catch(   ... ) {
    cerr << "unknown exception caught"<< endl;
  }

}

