/***************************************************************************
 *   Copyright (C) 2005 by Andreas Pokorny                                 *
 *   andreas.pokorny@biozentrum.uni-wuerzburg.de                           *
 *                                                                         *
 *   This file is part of profdist and cbcanalyzer                         *
 *                                                                         *
 *   Both profdist and cbcanalyzer are free software; you can redistribute * 
 *   it and/or modify it under the terms of the GNU General Public License * 
 *   as published by the Free Software Foundation; either version 2 of the * 
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   Profdist and cbcanalyzer are distributed in the hope that it will be  *
 *   useful, but WITHOUT ANY WARRANTY; without even the implied warranty   *
 *   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PROFDIST_BOOTSTRAP_H_INCLUDED
#define PROFDIST_BOOTSTRAP_H_INCLUDED

#include "aligncode.h"
#include "profile.h"

namespace profdist { 
/**
 * @brief generates a bootstrap from source in destination
 * @param source Alignment source, is not modified by that function
 * @param destination the destination, 
 * @param extension defines a suffix for the sequence names of the destination object, 
 * Bootstraps other alignments from a source alignment 
 * by random picking of sequence entries with put back.
*/
template<typename Traits>
void bootstrap( AlignCode<Traits> const& source, AlignCode<Traits> & destination, std::string const& extension ="" );

/**
 * @brief generates a bootstrap from source in destination
 * @param source profile source, is not modified by that function
 * @param destination the destination, 
 * Bootstraps other alignments from a source alignment 
 * by random picking of sequence entries with put back.
*/
void bootstrap( Profile const& source, Profile & destination );


template<typename Traits>
void bootstrap( AlignCode<Traits> const& s, AlignCode<Traits> & d, std::string const& extension )
{
  size_t num_entries = s.get_num_sites();

  std::vector<std::size_t> random_entries( num_entries, 0 );
  // Generate random numbers,
  for ( size_t i = random_entries.size(); i ; --i )
    ++random_entries[size_t( (float(random_entries.size()) *rand())/float(RAND_MAX+1.0f)) ];

  size_t num_seq = s.get_num_sequences();
  if( ! num_seq ) 
    return;

  // Generate sequence iterators on source, to read all data simultaneously
  typedef vector<typename AlignCode<Traits>::const_sequence_iterator> iterators_vec; 
  iterators_vec iterators( num_seq, s.begin(0) ); 
  
  size_t i = 0;
  for( typename iterators_vec::iterator it_it = iterators.begin(), 
      it_e = iterators.end(); it_it != it_e; ++it_it, ++i )
    *it_it = s.begin( i );

  vector<typename AlignCode<Traits>::element_type> items( num_seq );

  // Fast resize destination!
  d.resize( num_seq, 0 );


  for( std::size_t i = 0; i < num_seq; ++i ) 
    d.set_sequence_name( s.get_sequence_name(i) + extension, i );

  // Fill destination column by column using push_back
  for( vector<std::size_t >::iterator it = random_entries.begin(),
      e = random_entries.end(); it != e; ++it )
  {
    if( *it )
    {
      typename vector<typename AlignCode<Traits>::element_type>::iterator items_it = items.begin();
      for( 
          typename iterators_vec::iterator it_it = iterators.begin(), it_e = iterators.end(); 
          it_it != it_e; 
          ++it_it, ++items_it 
          )
      {
        *items_it = **it_it;
      }

      for( size_t i = 0; i < *it; ++i )
        d.push_back( items );
 
    }
    
    for( typename iterators_vec::iterator it_it = iterators.begin(), 
        it_e = iterators.end(); it_it != it_e; ++it_it )
      ++(*it_it); 
  }
}

}

#endif

