#ifndef PROFDIST_QTGUI_WORKSPACE_HPP_
#define PROFDIST_QTGUI_WORKSPACE_HPP_

#include <QObject>
#include <QSignalMapper>
#include <QMutex>
#include <QList>
#include <QIcon>

#include <vector>
#include <list>
#include <map>
#include <string>
#include <limits>

#include "operation_interface.hpp"
#include "data_type.hpp"

class QAction;
class QString;
class QIcon;

namespace profdist {
	class alignment;
}

namespace gui {
	
	class MainWindow;
	class Feature;
	class WorkspaceTreeModel;
	class WorkspaceTreeView;
	class TaskManager;
	class FileType;
	class GuiHandle;
	class GuiHandleData;
	class Viewer;
	class ViewerFactory;
	
	class Workspace : public QObject {
			Q_OBJECT
		public:
			Workspace(MainWindow& mainWindow);
			~Workspace();
			
			MainWindow* mainWindow();
			const MainWindow* mainWindow() const;
			
			void installNewFeature(Feature* feature, OperationInterface* operation);
			void installNewFileType(FileType* fileType);
			
			unsigned int getNewGuiHandleId();
			int installNewGuiHandle(GuiHandle* guiHandle); ///< Installs a new GuiHandler. If the GuiHandler could be installed a positive number is returned. If not a negative one is returned. This number is the handle identifier.
			
			void resumeTask(unsigned int taskId);
			void cancelTask(unsigned int taskId);
			
			bool viewerInstalled(unsigned int type);
			Viewer* getViewer(unsigned int type,
				const QString& name = "",
				const QIcon& icon = QIcon(":/images/alignment.png"),
				QWidget* parent = 0);
			void installViewer(ViewerFactory* factory);
			
			const std::list<FileType*> getFileTypes(const std::string& suffix);
			const std::list<FileType*> getFileTypes(const QString& suffix);
			
			QString getFileTypeFilter() const;
			
			DataItem* getSelectedItem();
			
			bool hasRunningTasks();
			
			std::list<Feature*> getFeaturesForSelectedItem();
			
			void loadPlugins();
		
		public Q_SLOTS:
			void addDataItem(DataItem* newItem, DataItem* parentItem = 0);
			void removeDataItem(DataItem* item);
			
			void triggerOperation(int index);
			void removeTask(Task* task);
			
			void open();
			void remove();
			void rename();
			void saveAs();
			void view();
			void settings();
			
			void showErrorMessage(const std::string& message);
			void showErrorMessage(const QString& message);
			
			void balloonMessage(const QString& message);
			void balloonMessage(const QString& title, const QString& message);
			
			void useGuiHandle(int handleId, unsigned int taskId,  GuiHandleData* data);
			
		protected:
			const FileType* chooseFileType(const std::list<FileType*>& fileTypes, const QString& fileName);
			
		private:
			MainWindow			*_main_window;
			std::list<Feature*>	_features; // these are the important data operations.
			std::list<QAction*>	_additional_actions; // are implemented by plug-ins for example.
			
			std::map<unsigned int, Viewer*> _viewers; // viewers for the different data types
			
			std::vector<OperationInterface*> _data_operations; // These are the registered data operation objects.
			QSignalMapper		_signal_mapper;
			int					_mapping_counter;
			
			unsigned int		_actual_gui_handle_id;
			std::vector<GuiHandle*>	_gui_handles;
			QMutex				_mutex_use_gui_handle;
			
			struct GuiHandleEntry {
				GuiHandle* handle;
				unsigned int taskId;
				GuiHandleData* data;
			};
			
			std::list<GuiHandleEntry>	_active_gui_handles;
			
			WorkspaceTreeModel	*_tree_model;
			WorkspaceTreeView	*_tree_view;
			TaskManager			*_task_manager;
			
			std::list<FileType*>	_file_types;
	};
	
}

#endif // PROFDIST_QTGUI_WORKSPACE_HPP_

