#include "tree_item.hpp"

#include <iostream>
#include <algorithm>

namespace gui {
	
	/*
	 * Some help structures for stl algorithms
	 */
	template<typename T>
	struct FindById {
		int id;
		
		FindById(int id) : id(id) {}
		
		bool operator()(T item)
		{
			return item->id() == id;
		}
	};
	
	template<typename T>
	struct ChildDeleter {
		void operator()(T* t)
		{
			if(t)
			{
				// We need to set the parent to 0 because of segfaults.
				t->_parent = 0;
				delete t;
			}
		}
	};
	
	
	/*
	 * TreeItem implementation.
	 */
	template<typename T>
	TreeItem<T>::TreeItem()
	: _id(getNewId()),
	_children(),
	_parent(0)
	{
		TreeItem::addPendingItem(static_cast<T*>(this));
	}
	
	template<typename T>
	TreeItem<T>::~TreeItem()
	{
		std::for_each(_children.begin(), _children.end(), ChildDeleter<T>());
		_children.clear();
		if(_parent)
			_parent->removeChild(static_cast<T*>(this));
		TreeItem<T>::removeItemFromObserver(static_cast<T*>(this));
	}
	
	template<typename T>
	void TreeItem<T>::addChild(T* child)
	{
		if(!child /*|| findRecursive(child->id())*/)
			return;
		_children.push_back(child);
		if(child->_parent)
			child->_parent->removeChild(child);
		child->_parent = static_cast<T*>(this);
		switchToBound(child);
	}
	
	template<typename T>
	void TreeItem<T>::removeChild(T* child)
	{
		if(!child || child->_parent != this)
			return;
		typename TreeItemList::iterator i = std::find(_children.begin(), _children.end(), child);
		if(i == _children.end())
			return;
		_children.erase(i);
		child->_parent = 0;
		switchToPending(child);
	}
	
	template<typename T>
	T* TreeItem<T>::parent() const
	{
		return _parent;
	}
	
	template<typename T>
	T* TreeItem<T>::child(size_t index)
	{
		typename TreeItemList::iterator i = _children.begin(), e = _children.end();
		for(; i != e && index > 0; i++, index--); //while(i++ != e && index--) ;
		return (i == e ? 0 : *i);
	}
	
	template<typename T>
	int TreeItem<T>::id() const
	{
		return _id;
	}
	
	template<typename T>
	T* TreeItem<T>::find(int id)
	{
		if(id == _id)
			return this;
		typename TreeItemList::const_iterator i = std::find_if(_children.begin(), _children.end(), FindById<T*>(id));
		return (i == _children.end() ? 0 : *i);
	}
	
	template<typename T>
	T* TreeItem<T>::findRecursive(int id)
	{
		if(id == _id)
			return static_cast<T*>(this);
		
		for(typename TreeItemList::const_iterator i = _children.begin(), e = _children.end(); i != e; i++)
		{
			T* item = (*i)->findRecursive(id);
			if(item)
				return item;
		}
		
		return 0;
	}
	
	template<typename T>
	size_t TreeItem<T>::childCount() const
	{
		return _children.size();
	}
	
	template<typename T>
	int TreeItem<T>::indexOf(const T* child) const
	{	
		int index = 0;
		
		for(typename TreeItemList::const_iterator i = _children.begin(), e = _children.end(); i != e; i++, index++)
			if(*i == child)
				return index;
		
		return -1;
	}
	
	template<typename T>
	size_t TreeItem<T>::index() const
	{
		return (_parent ? static_cast<size_t>(_parent->indexOf(static_cast<const T*>(this))) : 0);
	}
	
	template<typename T>
	typename TreeItem<T>::TreeItemList TreeItem<T>::getPendingItems()
	{
		return _pending_items;
	}
	
	template<typename T>
	typename TreeItem<T>::TreeItemList TreeItem<T>::getBoundItems()
	{
		return _bound_items;
	}
	
	template<typename T>
	int TreeItem<T>::getNewId()
	{
		static int idCounter = IdCounterStart;
		return idCounter++;
	}
	
	template<typename T>
	void TreeItem<T>::addPendingItem(T* item)
	{
		typename TreeItemList::const_iterator i =
			std::find(_pending_items.begin(), _pending_items.end(), item);
		if(i != _pending_items.end())
			return;
		_pending_items.push_back(item);
	}
	
	template<typename T>
	void TreeItem<T>::switchToBound(T* item)
	{
		_pending_items.remove(item);
		_bound_items.push_back(item);
	}
	
	template<typename T>
	void TreeItem<T>::switchToPending(T* item)
	{
		_bound_items.remove(item);
		_pending_items.push_back(item);
	}
	
	template<typename T>
	void TreeItem<T>::removeItemFromObserver(T* item)
	{
		_pending_items.remove(item);
		_bound_items.remove(item);
	}
	
	template<typename T>
	typename TreeItem<T>::TreeItemList TreeItem<T>::_pending_items;
	
	template<typename T>
	typename TreeItem<T>::TreeItemList TreeItem<T>::_bound_items;
	
}

