#include "task_manager.hpp"
#include "workspace.hpp"
#include "operation_interface.hpp"
#include "debug.hpp"
#include "data_item.hpp"
#include "main_window.hpp"
#include "gui_handle_data.hpp"

#include <QMutexLocker>

namespace gui {
	
	TaskManager::TaskManager(Workspace* workspace)
	: _workspace(workspace)
	{}
	
	TaskManager::~TaskManager()
	{}
	
	void TaskManager::addTask(OperationInterface* operation, bool showProgress)
	{
		
		if(!operation)
			return;
		
		Task* task = new Task(operation);
		Item item;
		item.task.reset(task);
		
		connect(task, SIGNAL(taskFinished(unsigned int)), this, SLOT(removeTask(unsigned int)));
		connect(task, SIGNAL(sigAddDataItem(DataItem*, DataItem*)), mainWindow->workspace(), SLOT(addDataItem(DataItem*, DataItem*)));
		connect(task, SIGNAL(sigShowErrorMessage(const QString&)), mainWindow->workspace(), SLOT(showErrorMessage(const QString&)));
		connect(task, SIGNAL(sigUseGuiHandle(int, unsigned int, GuiHandleData*)), mainWindow->workspace(), SLOT(useGuiHandle(int, unsigned int, GuiHandleData*)));
		
		if(!operation->preExecution())
		{
			removeTask(task->taskId());
			return;
		}
		
		if(showProgress && mainWindow)
		{
			mainWindow->trackProgressOfTask(task);
		}
		
		{
			QMutexLocker locker(&_mutex_tasks);
			_tasks[task->taskId()] = item;
		}
		
		task->start();
	}
	
	bool TaskManager::hasRunningTasks() const
	{
		return !_tasks.empty();
	}
	
	void TaskManager::resumeTask(unsigned int taskId)
	{
		std::map<unsigned int, Item>::iterator i = _tasks.find(taskId);
		
		if(i == _tasks.end())
			return;
		
		i->second.task->resume();
	}
	
	void TaskManager::removeTask(unsigned int taskId)
	{
		QMutexLocker locker(&_mutex_tasks);
		
		std::map<unsigned int, Item>::iterator i = _tasks.find(taskId);
		
		if(i == _tasks.end())
			return;
		
		_tasks.erase(i);
	}
	
}
