#ifndef PROFDIST_QTGUI_TASK_HPP_
#define PROFDIST_QTGUI_TASK_HPP_

#include <list>
#include <boost/shared_ptr.hpp>
#include <QThread>
#include <QMutex>
#include <QWaitCondition>

class QString;

namespace gui {
	
	class OperationInterface;
	class DataItem;
	class GuiHandleData;
	
	/**
	 * Specifies how long the task waits until it performs an hard terminate
	 * on the operation after calling cancel() on that operation.
	 */
	const unsigned int TaskCancelTimeout = 2000;
	
	
	/**
	 * Basic class for starting tasks in the profdist environment. This tasks
	 * expect an AbstractOperation object which they execute in their own
	 * thread context. So profdist can compute more than on operation at once.
	 * 
	 * The tasks are managed by the TaskManager class. The TaskManager is
	 * responsible for removing the task from the system and to free its
	 * resources.
	 */
	class Task : protected QThread /*, public Task*/ {
			Q_OBJECT
		public:
			/**
			 * Constructor builds a new task object and is initialized by the
			 * passed AbstractOperation object.
			 * @param operation Operation the Task is initialized with. When
			 * the user calls start on this task the operation's run() method
			 * is executed.
			 */
			Task(OperationInterface* operation);
			
			/**
			 * Destructor. Cancels the execution context and waits until the
			 * thread returns.
			 */
			~Task();
			
			/**
			 * @return Returns the task's taskId.
			 */
			unsigned int taskId() const;
			
			/**
			 * Starts execution of this task.
			 */
			void start();
			
			
			/**
			 * Same as posix join.
			 */
			bool wait(unsigned long time = ULONG_MAX);
			
			/**
			 * Test if the user has canceled the executiion of this task. The
			 * AbstractOperation object calls this method do determine if it
			 * should continue or cancel exection.
			 * @return Returns true if the user canceled this task, otherwise
			 * false;
			 */
			bool isCanceled() const;
			
			/**
			 * Returns the associated operation object as a smart pointer.
			 */
			boost::shared_ptr<OperationInterface> operation();
			
			/**
			 * Pauses execution of thread and waits until resumeTask() is
			 * called. This function should only be called by a OperationInterface
			 * object belonging to this Task object.
			 * @param section Defines the section in which the thread is
			 * currently paused. This value is passed to the main window. There
			 * it resolves the approbiate Gui-Response (that means it shows the
			 * right dialog for the thread) and displays the dialog. From this
			 * dialog the expected values are taken to the source thread so it
			 * can continue processing with the new data.
			 */
			void pause(int guiHandleId, GuiHandleData* data);
			
			/**
			 * Sets the maximum value for the progress tracking.
			 */
			void setMaximumValue(size_t maximumValue);
			
			/**
			 * Updates the progress value by value.
			 */
			void updateProgress(size_t value);
			
			/**
			 * Updates the progress message by message.
			 */
			void updateProgress(const QString& message);
			
			/**
			 * Adds a new node to the workspace which has parentNode as parent.
			 */
			void addDataItem(DataItem* newNode, DataItem* parentNode = 0);
			
			/**
			 * Shows an error message. This method is used indirectly by other
			 * threads. This method then emits an signal with the passed message
			 * which is received by the main thread to show it by he gui.
			 */
			void showErrorMessage(const QString& message);
			
			
		public Q_SLOTS:
			/**
			 * Resumes a paused task.
			 */
			void resume();
			
			/**
			 * Cancels execution of this task.
			 */
			void cancel();
			
		Q_SIGNALS:
			/**
			 * Signal is emitted when the task finished. This signal is also
			 * emitted if the task was canceled or the task aborted. The
			 * taskPtr identifies the task os it can be deleted from the list
			 * of running tasks.
			 */
			void taskFinished(unsigned int taskId);
			
			/**
			 * Signals that the task paused.
			 */
			void taskPaused(unsigned int taskId);
			
			/**
			 * Signal to set the maximum value.
			 */
			void sigSetMaximumValue(unsigned int maximumValue);
			
			/**
			 * Signal to update the progress status.
			 */
			void sigUpdateProgress(unsigned int value);
			
			/**
			 * Signal to update the progress status.
			 */
			void sigUpdateProgress(const QString& message);
			
			/**
			 * Signal to add a new node.
			 */
			void sigAddDataItem(DataItem* newNode, DataItem* parentNode);
			
			/**
			 * Signal to show an error message.
			 */
			void sigShowErrorMessage(const QString& message);
			
			void sigUseGuiHandle(int guiHandleId, unsigned int taskId, GuiHandleData* data);
			
		protected:
			/**
			 * Executes the OperationInterface::execute() method. 
			 */
			void run();
			
		private:
			/*
			 * Don't allow copies.
			 */
			Task(const Task& t);
			Task& operator=(const Task& t);
			
			
			const unsigned int						_task_id;
			bool									_is_canceled;
			boost::shared_ptr<OperationInterface>	_operation;
			
			QMutex									_mutex_condition;
			QWaitCondition							_pause_condition;
			
			bool									_ended_regular;
	};
	
}

#endif // PROFDIST_QTGUI_TASK_HPP_

