#include "profile_selection_dialog.hpp"
#include "types.h"
#include "debug.hpp"
#include "config.h"
#include <QTreeWidget>
#include <QFileDialog>
#include <QList>
#include <QMessageBox>
#include <QPushButton>
#include <QHeaderView>
#include <QSettings>
#include <algorithm>

namespace gui {
	
	QString createTreeNodeString(const QString& name, size_t sequence_count)
	{
		return name + " [" + QString().setNum(sequence_count) + "]";
	}
	
	QString createTreeNodeString(const std::string& name, size_t sequence_count)
	{
		return QString(name.c_str()) + " [" + QString().setNum(sequence_count) + "]";
	}
	
	ProfileTreeNode::ProfileTreeNode(const QString& name, int id, size_t sequence_count)
	: TreeNode(name, id, sequence_count)
	{
		Qt::ItemFlags flag = flags();
		flag |= Qt::ItemIsEditable;
		setFlags(flag);
	}
	
	ProfileTreeNode::ProfileTreeNode(const std::string& name, int id, size_t sequence_count)
	: TreeNode(name, id, sequence_count)
	{
		Qt::ItemFlags flag = flags();
		flag |= Qt::ItemIsEditable;
		setFlags(flag);
	}
	
	TreeNode::TreeNode(const QString& name, int id, size_t sequence_count)
	: _name(name),
	_id(id),
	_sequence_count(sequence_count)
	{
		if(id >= 0)
			setText(0, name);
		else
			setText(0, createTreeNodeString(name, sequence_count));
	}
	
	TreeNode::TreeNode(const std::string& name, int id, size_t sequence_count)
	: _name(QString(name.c_str())),
	_id(id),
	_sequence_count(sequence_count)
	{
		if(id >= 0)
			setText(0, QString(name.c_str()));
		else
			setText(0, createTreeNodeString(name, sequence_count));
	}
	
	void TreeNode::incrementSequenceCount(int step)
	{
		_sequence_count += step;
		setText(0, createTreeNodeString(_name, _sequence_count));
	}
	
	QString TreeNode::getName() const
	{
		return _name;
	}
	
	int TreeNode::getId() const
	{
		return _id;
	}
	
	size_t TreeNode::getSequenceCount() const
	{
		return _sequence_count;
	}
	
	void TreeNode::clear()
	{
		QList<QTreeWidgetItem*> children = takeChildren();
		foreach(QTreeWidgetItem* item, children)
		{
			delete item;
		}
		
		incrementSequenceCount(-_sequence_count);
	}
	
	/*
	 * ProfileSelectionDialog:
	 */
	
	ProfileSelectionDialog::ProfileSelectionDialog(
		const sequences_t& sequences,
		const classification_t& class_data,
		bool noKimura,
		QWidget* parent)
	: QDialog(parent),
	_sequences(sequences),
	_class_data(class_data),
	_profile_counter(1),
	_sequences_root(0),
	_profiles_root(0),
	_no_kimura(noKimura)
	{
		setupUi(this);
		
		// build the remaining profiles by default:
		_check_box_build_remaining->setCheckState(Qt::Checked);
		
		if(_no_kimura)
			_combo_box_correction_model->removeItem(1); // remove kimura
		
		_sequence_tree_widget->header()->setStretchLastSection(false);
		_sequence_tree_widget->header()->setResizeMode(QHeaderView::ResizeToContents); 
		
		_profile_tree_widget->header()->setStretchLastSection(false);
		_profile_tree_widget->header()->setResizeMode(QHeaderView::ResizeToContents); 
		
		_sequences_root = new TreeNode(tr("Sequences"), -1);
		//_sequences_root->setText(0, tr("Sequences"));
		_sequences_root->setFlags(_sequences_root->flags() & ~Qt::ItemIsSelectable);
		_sequence_tree_widget->addTopLevelItem(_sequences_root);
		
		_profiles_root = new TreeNode(tr("Profiles"), -1);
		//_profiles_root->setText(0, tr("Profiles"));
		_profiles_root->setFlags(_profiles_root->flags() & ~Qt::ItemIsSelectable);
		_profile_tree_widget->addTopLevelItem(_profiles_root);
		
		_widget_expert_mode->setVisible(false);
		
		connect(_combo_box_correction_model, SIGNAL(currentIndexChanged(int)), this, SLOT(correctionModelSelectionChanged(int)));
		connect(_tool_button_open_file, SIGNAL(clicked()), this, SLOT(getRateMatrixFile()));
		connect(_reset_profiles_button, SIGNAL(clicked()), this, SLOT(resetProfiles()));
		connect(_add_profile_button, SIGNAL(clicked()), this, SLOT(createProfile()));
		connect(_profile_tree_widget, SIGNAL(itemClicked(QTreeWidgetItem*, int)), this, SLOT(editItem(QTreeWidgetItem*, int)));
		
		QPushButton* okButton = buttonBox->button(QDialogButtonBox::Ok);
		connect(okButton, SIGNAL(clicked()), this, SLOT(testProfiles()));
		
		_sequence_tree_widget->setSelectionMode(QAbstractItemView::ExtendedSelection);
		
		setupSequencesTree(_sequences, _class_data);
	}
	
	void ProfileSelectionDialog::setupSequencesTree(const sequences_t& sequences,
		const classification_t& class_data)
	{
		size_t count = 0;
		
		/*
		 * New implementation of setupSeqeuenceTree:
		 */ 
		typedef std::list<std::string> classification_list_t;
		
		classification_t::const_iterator class_iter = class_data.begin(),
			class_iter_end = class_data.end();
		
		/*
		 * Run through all sequences of the alignment
		 */
		for(sequences_t::const_iterator i = sequences.begin(), e = sequences.end();
			i != e; i++, class_iter++, count++)
		{
			/*
			 * Get the begin and end iterator of the classification list of the
			 * actual sequence.
			 */
			classification_list_t::const_iterator class_list_iter = class_iter->begin(),
				class_list_iter_end = class_iter->end();
			
			/*
			 * Stores the actual insertion point
			 */
			TreeNode* insertion_point = _sequences_root;
			
			/*
			 * Iterate through all classifications of this sequence
			 */
			for(; class_list_iter != class_list_iter_end && insertion_point->childCount() > 0; class_list_iter++)
			{
				bool has_class = false;
				
				for(int index = 0; index < insertion_point->childCount(); index++)
				{
					TreeNode* tmp_item = static_cast<TreeNode*>(insertion_point->child(index));
					
					if(tmp_item->getName() == QString(class_list_iter->c_str()))
					{
						insertion_point->incrementSequenceCount();
						has_class = true;
						insertion_point = tmp_item;
						break;
					}
				}
				
				if(!has_class)
				{
					insertion_point->incrementSequenceCount();
					TreeNode* class_node = new TreeNode(*class_list_iter, -1, 0);
					insertion_point->addChild(class_node);
					insertion_point = class_node;
				}
			}
			
			for(; class_list_iter != class_list_iter_end; class_list_iter++)
			{
				insertion_point->incrementSequenceCount();
				TreeNode* class_node = new TreeNode(*class_list_iter, -1, 0);
				insertion_point->addChild(class_node);
				insertion_point = class_node;
			}
			
			insertion_point->incrementSequenceCount();
			TreeNode* leave = new TreeNode(*i, count, 1);
			insertion_point->addChild(leave);
		}
	}
	
	void ProfileSelectionDialog::getProfiles(profile_list_t& profiles, profile_names_t& profile_names, profile_sequence_names_t& profile_sequence_names)
	{	
		if(!profile_names.empty())
			profile_names.clear();
		
		size_t numProfiles = _profiles_root->childCount();
		
		if(_check_box_build_remaining->isChecked())
			numProfiles += _sequences_root->getSequenceCount();
		
		QList<QTreeWidgetItem*> profile_items = _profiles_root->takeChildren();
		
		foreach(QTreeWidgetItem* i, profile_items)
		{
			TreeNode* item = static_cast<TreeNode*>(i);
			std::list<size_t> sequence_numbers;
			std::list<std::string> sequence_names;
			
			/*
			 * Collect from the actual node all the names and identifiers of
			 * the sequences which are in this profile.
			 */
			getSingleProfile(item, sequence_numbers, sequence_names);
			
			profiles.push_back(sequence_numbers);
			profile_names.push_back(item->text(0).toStdString());
			profile_sequence_names.push_back(sequence_names);
		}
		
		if(_check_box_build_remaining->isChecked())
		{
			QList<QTreeWidgetItem*> sequence_items = _sequences_root->takeChildren();
		
			foreach(QTreeWidgetItem* i, sequence_items)
			{
				TreeNode* item = static_cast<TreeNode*>(i);
				
				if(item->getId() >= 0)
				{
					// we reached a node with a sequence
					
					std::list<size_t> sequence_numbers;
					std::list<std::string> sequence_names;
				
					getSingleProfile(item, sequence_numbers, sequence_names);
			
					profiles.push_back(sequence_numbers);
					profile_names.push_back(item->text(0).toStdString());
					profile_sequence_names.push_back(sequence_names);
				}
				else
				{
					// we reached a classification node
					traverseClassificationNode(item, profiles, profile_names, profile_sequence_names);
				}
			}
		}
	}
	
	void ProfileSelectionDialog::getSingleProfile(
		TreeNode* item,
		std::list<size_t>& sequence_numbers,
		std::list<std::string>& sequence_names)
	{
		if(!item)
			return;
		
		/*
		 * We reached a leave, so add it as a sequence to the profile.
		 */
		if(item->childCount() == 0 && item->getId() >= 0)
		{
			sequence_numbers.push_back(static_cast<size_t>(item->getId()));
			sequence_names.push_back(item->getName().toStdString());
		}
		/*
		 * otherwise we reached an inner node, so we have to go on.
		 */
		else
		{
			QList<QTreeWidgetItem*> children = item->takeChildren();
			
			foreach(QTreeWidgetItem* i, children)
			{
				TreeNode* item = static_cast<TreeNode*>(i);
				
				getSingleProfile(item, sequence_numbers, sequence_names);
			} 
		}
	}
	
	void ProfileSelectionDialog::traverseClassificationNode(TreeNode* item,
		profile_list_t& profiles,
		profile_names_t& profile_names,
		profile_sequence_names_t& profile_sequence_names)
	{
		QList<QTreeWidgetItem*> children = item->takeChildren();
		
		foreach(QTreeWidgetItem* i, children)
		{
			TreeNode* child = static_cast<TreeNode*>(i);
			if(child->getId() == -1)
				traverseClassificationNode(child, profiles, profile_names, profile_sequence_names);
			else
			{
				std::list<size_t> sequence_numbers;
				sequence_numbers.push_back(child->getId());
				
				std::list<std::string> sequence_names;
				sequence_names.push_back(child->getName().toStdString());
				
				profiles.push_back(sequence_numbers);
				profile_names.push_back(child->getName().toStdString());
				profile_sequence_names.push_back(sequence_names);
			}
		}
	}
	
	profdist::CorrectionModel ProfileSelectionDialog::correctionModel() const
	{
		int cm = _combo_box_correction_model->currentIndex();
		if(_no_kimura && cm > 0)
			cm += 1;
		return profdist::CorrectionModel(cm);
	}
		
	void ProfileSelectionDialog::correctionModelSelectionChanged(int index)
	{
		int offset = _no_kimura ? 1 : 0;
		if(index < (profdist::GTR - offset))
		{
			_line_edit_rate_matrix->setEnabled(false);
			_tool_button_open_file->setEnabled(false);
		}
		else
		{
			_line_edit_rate_matrix->setEnabled(true);
			_tool_button_open_file->setEnabled(true);
		}
	}
	
	void ProfileSelectionDialog::createProfile()
	{
		QList<QTreeWidgetItem*> selectedItems = _sequence_tree_widget->selectedItems();
		
		if(selectedItems.size() < 1)
			return;
		
		int sequences_sum = 0;
		
		foreach(QTreeWidgetItem* item, selectedItems)
		{
			TreeNode* ti = static_cast<TreeNode*>(item);
			sequences_sum += ti->getSequenceCount();
		}
		
		size_t profileNumber = _profile_counter++;
		QString name = "Profile " + QString().setNum(profileNumber) + " [" + QString().setNum(sequences_sum) + "]";
		ProfileTreeNode* profile = new ProfileTreeNode(name.toStdString(), profileNumber);
		_profiles_root->addChild(profile);
		_profiles_root->setExpanded(true);
		
		foreach(QTreeWidgetItem* item, selectedItems)
		{
			QTreeWidgetItem* parent = item->parent();
			int index = parent->indexOfChild(item);
			parent->takeChild(index);
			size_t reducedSequences = static_cast<TreeNode*>(item)->getSequenceCount();
			static_cast<TreeNode*>(parent)->incrementSequenceCount(-(reducedSequences));
			profile->addChild(item);
			
			/*
			 * Decrementing the sequence count in all parent nodes
			 */
			{
				QTreeWidgetItem* parent_1 = parent, *grand_parent_1 = parent_1->parent();
				while(grand_parent_1)
				{
					parent_1 = grand_parent_1;
					static_cast<TreeNode*>(parent_1)->incrementSequenceCount(-(reducedSequences));
					grand_parent_1 = grand_parent_1->parent();
				}
			}
			/*
			 * Delete all parent nodes which only have the path to this item as
			 * their only child. So this child will be deleted and the parent
			 * node is useless.
			 */
			if(parent->childCount() < 1)
			{
				QTreeWidgetItem* grand_parent = parent->parent();
				while(grand_parent && grand_parent != _sequences_root && grand_parent->childCount() < 2)
				{
					parent = grand_parent;
					grand_parent = grand_parent->parent();
				}
				
				delete parent;
			}
		}
		
		_profiles_root->setText(0, createTreeNodeString(QString("Profiles "), _profiles_root->childCount()));
	}
	
	void ProfileSelectionDialog::resetProfiles()
	{
		_profiles_root->clear();
		_sequences_root->clear();
		
		setupSequencesTree(_sequences, _class_data);
		_profile_counter = 1;
	}
	
	void ProfileSelectionDialog::getRateMatrixFile()
	{
		QSettings settings;
		QString dir = settings.value("settings/rate_matrix_open_path").toString();
		if(dir.isNull())
#ifdef Q_OS_WIN32
			dir = QCoreApplication::applicationDirPath();
#else
			dir = PROFDIST_DATA_DIR "/data";
#endif
		QString fileName = QFileDialog::getOpenFileName(this, tr("Open Rate Matrix"), dir);
		if(fileName == "")
			return;
		_line_edit_rate_matrix->setText(fileName);
	}
	
	void ProfileSelectionDialog::testProfiles()
	{
		// The user has to define at least 3 profiles if no profiles are build
		// for the rest of the sequences.
		if(_check_box_build_remaining->isChecked())
			accept();
		else
		{
			if(_profiles_root->childCount() >= 3)
				accept();
			else
			{
				QMessageBox::information(this,
					tr("Profile definition"),
					tr("You have to define at least 3 profiles\nso Profdist can continue Profile Neighbour Joining."));
			}
		}
	}
	
	void ProfileSelectionDialog::editItem(QTreeWidgetItem* item, int column)
	{
		if(!item)
			return;
		if(dynamic_cast<ProfileTreeNode*>(item))
			_profile_tree_widget->editItem(item, column);
	}
	
	/*
	 * SmallProfileSelectionDialog:
	 */
	SmallProfileSelectionDialog::SmallProfileSelectionDialog(
		const sequences_t& sequences,
		const classification_t& class_data,
		QWidget* parent)
	: QDialog(parent),
	_sequences(sequences),
	_class_data(class_data),
	_profile_counter(1),
	_sequences_root(0),
	_profiles_root(0)
	{
		setupUi(this);
		
		_sequence_tree_widget->header()->setStretchLastSection(false);
		_sequence_tree_widget->header()->setResizeMode(QHeaderView::ResizeToContents); 
		
		_profile_tree_widget->header()->setStretchLastSection(false);
		_profile_tree_widget->header()->setResizeMode(QHeaderView::ResizeToContents); 
		
		_sequences_root = new TreeNode(tr("Sequences"), -1);
		//_sequences_root->setText(0, tr("Sequences"));
		_sequences_root->setFlags(_sequences_root->flags() & ~Qt::ItemIsSelectable);
		_sequence_tree_widget->addTopLevelItem(_sequences_root);
		
		_profiles_root = new TreeNode(tr("Profiles"), -1);
		//_profiles_root->setText(0, tr("Profiles"));
		_profiles_root->setFlags(_profiles_root->flags() & ~Qt::ItemIsSelectable);
		_profile_tree_widget->addTopLevelItem(_profiles_root);
		
		connect(_reset_profiles_button, SIGNAL(clicked()), this, SLOT(resetProfiles()));
		connect(_add_profile_button, SIGNAL(clicked()), this, SLOT(createProfile()));
		connect(_profile_tree_widget, SIGNAL(itemClicked(QTreeWidgetItem*, int)), this, SLOT(editItem(QTreeWidgetItem*, int)));
		
		QPushButton* okButton = buttonBox->button(QDialogButtonBox::Ok);
		connect(okButton, SIGNAL(clicked()), this, SLOT(testProfiles()));
		
		_sequence_tree_widget->setSelectionMode(QAbstractItemView::ExtendedSelection);
		
		setupSequencesTree(_sequences, _class_data);
	}
	
	SmallProfileSelectionDialog::SmallProfileSelectionDialog(QWidget* parent)
	: QDialog(parent)
	{
		setupUi(this);
		
		_sequence_tree_widget->header()->setStretchLastSection(false);
		_sequence_tree_widget->header()->setResizeMode(QHeaderView::ResizeToContents); 
		
		_profile_tree_widget->header()->setStretchLastSection(false);
		_profile_tree_widget->header()->setResizeMode(QHeaderView::ResizeToContents); 
		
		_sequences_root = new TreeNode(tr("Sequences"), -1);
		//_sequences_root->setText(0, tr("Sequences"));
		_sequences_root->setFlags(_sequences_root->flags() & ~Qt::ItemIsSelectable);
		_sequence_tree_widget->addTopLevelItem(_sequences_root);
		
		_profiles_root = new TreeNode(tr("Profiles"), -1);
		//_profiles_root->setText(0, tr("Profiles"));
		_profiles_root->setFlags(_profiles_root->flags() & ~Qt::ItemIsSelectable);
		_profile_tree_widget->addTopLevelItem(_profiles_root);
		
		connect(_reset_profiles_button, SIGNAL(clicked()), this, SLOT(resetProfiles()));
		connect(_add_profile_button, SIGNAL(clicked()), this, SLOT(createProfile()));
		connect(_profile_tree_widget, SIGNAL(itemClicked(QTreeWidgetItem*, int)), this, SLOT(editItem(QTreeWidgetItem*, int)));
		
		QPushButton* okButton = buttonBox->button(QDialogButtonBox::Ok);
		connect(okButton, SIGNAL(clicked()), this, SLOT(testProfiles()));
		
		_sequence_tree_widget->setSelectionMode(QAbstractItemView::ExtendedSelection);
		
		//setupSequencesTree(_sequences, _class_data);
	}
	
	void SmallProfileSelectionDialog::setupSequencesTree(const sequences_t& sequences,
		const classification_t& class_data)
	{
		size_t count = 0;
		
		/*
		 * New implementation of setupSeqeuenceTree:
		 */ 
		typedef std::list<std::string> classification_list_t;
		
		classification_t::const_iterator class_iter = class_data.begin(),
			class_iter_end = class_data.end();
		
		/*
		 * Run through all sequences of the alignment
		 */
		for(sequences_t::const_iterator i = sequences.begin(), e = sequences.end();
			i != e; i++, class_iter++, count++)
		{
			/*
			 * Get the begin and end iterator of the classification list of the
			 * actual sequence.
			 */
			classification_list_t::const_iterator class_list_iter = class_iter->begin(),
				class_list_iter_end = class_iter->end();
			
			/*
			 * Stores the actual insertion point
			 */
			TreeNode* insertion_point = _sequences_root;
			
			/*
			 * Iterate through all classifications of this sequence
			 */
			for(; class_list_iter != class_list_iter_end && insertion_point->childCount() > 0; class_list_iter++)
			{
				bool has_class = false;
				
				for(int index = 0; index < insertion_point->childCount(); index++)
				{
					TreeNode* tmp_item = static_cast<TreeNode*>(insertion_point->child(index));
					
					if(tmp_item->getName() == QString(class_list_iter->c_str()))
					{
						insertion_point->incrementSequenceCount();
						has_class = true;
						insertion_point = tmp_item;
						break;
					}
				}
				
				if(!has_class)
				{
					insertion_point->incrementSequenceCount();
					TreeNode* class_node = new TreeNode(*class_list_iter, -1, 0);
					insertion_point->addChild(class_node);
					insertion_point = class_node;
				}
			}
			
			for(; class_list_iter != class_list_iter_end; class_list_iter++)
			{
				insertion_point->incrementSequenceCount();
				TreeNode* class_node = new TreeNode(*class_list_iter, -1, 0);
				insertion_point->addChild(class_node);
				insertion_point = class_node;
			}
			
			insertion_point->incrementSequenceCount();
			TreeNode* leave = new TreeNode(*i, count, 1);
			insertion_point->addChild(leave);
		}
	}
	
	void SmallProfileSelectionDialog::getProfiles(profile_list_t& profiles, profile_names_t& profile_names, profile_sequence_names_t& profile_sequence_names)
	{	
		if(!profile_names.empty())
			profile_names.clear();
		
		size_t numProfiles = _profiles_root->childCount();
		
		if(_check_box_build_remaining->isChecked())
			numProfiles += _sequences_root->getSequenceCount();
		
		QList<QTreeWidgetItem*> profile_items = _profiles_root->takeChildren();
		
		foreach(QTreeWidgetItem* i, profile_items)
		{
			TreeNode* item = static_cast<TreeNode*>(i);
			std::list<size_t> sequence_numbers;
			std::list<std::string> sequence_names;
			
			/*
			 * Collect from the actual node all the names and identifiers of
			 * the sequences which are in this profile.
			 */
			getSingleProfile(item, sequence_numbers, sequence_names);
			
			profiles.push_back(sequence_numbers);
			profile_names.push_back(item->text(0).toStdString());
			profile_sequence_names.push_back(sequence_names);
		}
		
		/*
		if(_check_box_build_remaining->isChecked())
		{
			QList<QTreeWidgetItem*> sequence_items = _sequences_root->takeChildren();
		
			foreach(QTreeWidgetItem* i, sequence_items)
			{
				TreeNode* item = static_cast<TreeNode*>(i);
				
				if(item->getId() >= 0)
				{
					// we reached a node with a sequence
					
					std::list<size_t> sequence_numbers;
					std::list<std::string> sequence_names;
				
					getSingleProfile(item, sequence_numbers, sequence_names);
			
					profiles.push_back(sequence_numbers);
					profile_names.push_back(item->text(0).toStdString());
					profile_sequence_names.push_back(sequence_names);
				}
				else
				{
					// we reached a classification node
					traverseClassificationNode(item, profiles, profile_names, profile_sequence_names);
				}
			}
		}
		*/
	}
	
	void SmallProfileSelectionDialog::getSingleProfile(
		TreeNode* item,
		std::list<size_t>& sequence_numbers,
		std::list<std::string>& sequence_names)
	{
		if(!item)
			return;
		
		/*
		 * We reached a leave, so add it as a sequence to the profile.
		 */
		if(item->childCount() == 0 && item->getId() >= 0)
		{
			sequence_numbers.push_back(static_cast<size_t>(item->getId()));
			sequence_names.push_back(item->getName().toStdString());
		}
		/*
		 * otherwise we reached an inner node, so we have to go on.
		 */
		else
		{
			QList<QTreeWidgetItem*> children = item->takeChildren();
			
			foreach(QTreeWidgetItem* i, children)
			{
				TreeNode* item = static_cast<TreeNode*>(i);
				
				getSingleProfile(item, sequence_numbers, sequence_names);
			} 
		}
	}
	
	void SmallProfileSelectionDialog::traverseClassificationNode(TreeNode* item,
		profile_list_t& profiles,
		profile_names_t& profile_names,
		profile_sequence_names_t& profile_sequence_names)
	{
		QList<QTreeWidgetItem*> children = item->takeChildren();
		
		foreach(QTreeWidgetItem* i, children)
		{
			TreeNode* child = static_cast<TreeNode*>(i);
			if(child->getId() == -1)
				traverseClassificationNode(child, profiles, profile_names, profile_sequence_names);
			else
			{
				std::list<size_t> sequence_numbers;
				sequence_numbers.push_back(child->getId());
				
				std::list<std::string> sequence_names;
				sequence_names.push_back(child->getName().toStdString());
				
				profiles.push_back(sequence_numbers);
				profile_names.push_back(child->getName().toStdString());
				profile_sequence_names.push_back(sequence_names);
			}
		}
	}
	
	void SmallProfileSelectionDialog::createProfile()
	{
		QList<QTreeWidgetItem*> selectedItems = _sequence_tree_widget->selectedItems();
		
		if(selectedItems.size() < 1)
			return;
		
		int sequences_sum = 0;
		
		foreach(QTreeWidgetItem* item, selectedItems)
		{
			TreeNode* ti = static_cast<TreeNode*>(item);
			sequences_sum += ti->getSequenceCount();
		}
		
		size_t profileNumber = _profile_counter++;
		QString name = "Profile " + QString().setNum(profileNumber) + " [" + QString().setNum(sequences_sum) + "]";
		ProfileTreeNode* profile = new ProfileTreeNode(name.toStdString(), profileNumber);
		_profiles_root->addChild(profile);
		_profiles_root->setExpanded(true);
		
		foreach(QTreeWidgetItem* item, selectedItems)
		{
			QTreeWidgetItem* parent = item->parent();
			int index = parent->indexOfChild(item);
			parent->takeChild(index);
			size_t reducedSequences = static_cast<TreeNode*>(item)->getSequenceCount();
			static_cast<TreeNode*>(parent)->incrementSequenceCount(-(reducedSequences));
			profile->addChild(item);
			
			/*
			 * Decrementing the sequence count in all parent nodes
			 */
			{
				QTreeWidgetItem* parent_1 = parent, *grand_parent_1 = parent_1->parent();
				while(grand_parent_1)
				{
					parent_1 = grand_parent_1;
					static_cast<TreeNode*>(parent_1)->incrementSequenceCount(-(reducedSequences));
					grand_parent_1 = grand_parent_1->parent();
				}
			}
			/*
			 * Delete all parent nodes which only have the path to this item as
			 * their only child. So this child will be deleted and the parent
			 * node is useless.
			 */
			if(parent->childCount() < 1)
			{
				QTreeWidgetItem* grand_parent = parent->parent();
				while(grand_parent && grand_parent != _sequences_root && grand_parent->childCount() < 2)
				{
					parent = grand_parent;
					grand_parent = grand_parent->parent();
				}
				
				delete parent;
			}
		}
		
		_profiles_root->setText(0, createTreeNodeString(QString("Profiles "), _profiles_root->childCount()));
	}
	
	void SmallProfileSelectionDialog::resetProfiles()
	{
		_profiles_root->clear();
		_sequences_root->clear();
		
		setupSequencesTree(_sequences, _class_data);
		_profile_counter = 1;
	}
	
	void SmallProfileSelectionDialog::testProfiles()
	{
		// The user has to define at least 3 profiles if no profiles are build
		// for the rest of the sequences.
		/*
		if(_check_box_build_remaining->isChecked())
			accept();
		else
		{
			if(_profiles_root->childCount() >= 3)
				accept();
			else
			{
				QMessageBox::information(this,
					tr("Profile definition"),
					tr("You have to define at least 3 profiles\nso Profdist can continue Profile Neighbour Joining."));
			}
		}
		*/
	}
	
	void SmallProfileSelectionDialog::editItem(QTreeWidgetItem* item, int column)
	{
		if(!item)
			return;
		if(dynamic_cast<ProfileTreeNode*>(item))
			_profile_tree_widget->editItem(item, column);
	}
	
}

