#ifndef PROFDIST_QTGUI_DISTANCE_OPERATION_HPP_
#define PROFDIST_QTGUI_DISTANCE_OPERATION_HPP_

#include "operation_interface.hpp"
#include "distance_dialog.hpp"
#include "operation_accessors.hpp"
#include "distance_data.hpp"
#include "main_window.hpp"
#include "progress_update.h"
#include "distance.h"

namespace gui {
	
	template<typename Traits>
	class DistanceOperation : public OperationInterface {
		public:
									DistanceOperation();
									~DistanceOperation();
			void					execute();
			bool					setDataItem(DataItem* item);
			OperationInterface*		clone();
			bool					preExecution();
		private:
			bool					loadRateMatrix(const QString& fileName, typename Traits::rate_matrix& matrix);
			
			DataForDistance<Traits>	*_data_for_distance;
			profdist::CorrectionModel	_correction;
			QString					_matrix_file_name;
	};
	
	template<typename Traits>
	DistanceOperation<Traits>::DistanceOperation()
	: _data_for_distance(0), _correction(profdist::Jukes), _matrix_file_name()
	{}
	
	template<typename Traits>
	DistanceOperation<Traits>::~DistanceOperation()
	{
		GUI_MSG_OBJ_DESTROYED(DistanceOperation);
	}
	
	template<typename Traits>
	void DistanceOperation<Traits>::execute()
	{
		try {
			setMaximumValue(_data_for_distance->numBootstraps());
			updateProgress(0);
			updateProgress(QObject::tr("Generating ") + QString().setNum(_data_for_distance->numBootstraps()) + QObject::tr(" distance matrices"));
			
			/*
			* Sequence names
			*/
			std::vector<std::string> sequenceNames = _data_for_distance->bootstrapData(0).get_sequence_names();
			
			typename Traits::rate_matrix Q(0);
			
			if(_correction > profdist::Kimura && !loadRateMatrix(_matrix_file_name, Q))
			{
				showErrorMessage(QObject::tr("Could not load rate matrix '") + _matrix_file_name + QObject::tr("'"));
				return;
			}
			
			/*
			* Distance Matrices
			*/
			int numSequences = _data_for_distance->bootstrapData(0).get_num_sequences();
			DistanceData::distance_matrices_t distanceMatrices(_data_for_distance->numBootstraps(), profdist::distance_matrix(numSequences, numSequences, 0.0));
			profdist::ProgressSink sink;
			
			try
			{
				size_t index = 0; // Index for iterating through the bootstraps.
				for(DistanceData::distance_matrices_t::iterator i = distanceMatrices.begin(), e = distanceMatrices.end(); i != e; i++, index++)
				{
					if(isCanceled())
						return;
					
					profdist::AlignCode<Traits> source = _data_for_distance->bootstrapData(index);
					profdist::compute_distance(source, *i, Q, _correction, profdist::Eigenvalue, profdist::NewtonMethod, sink);
					
					updateProgress(index);
				}
			}
			catch(const std::runtime_error& e)
			{
				showErrorMessage(QObject::tr("Error during distance operation:\n") + QString(e.what()));
				return;
			}
			catch(const error& e)
			{
				showErrorMessage(QObject::tr("Error during distance operation:\n") + QString(e));
				return;
			}
			catch(const std::exception& e)
			{
				showErrorMessage(QObject::tr("Error during distance operation:\n") + QString(e.what()));
				return;
			}
			
			DistanceData* dd = new DistanceData(_data_for_distance->name() + "_distance", sequenceNames, distanceMatrices);
			addDataItem(dd, _data_for_distance);
			_data_for_distance->decrementUseCount();
		}
		catch(const std::exception& e)
		{
			GUI_MSG_N("Error during distance operation:\n" << e.what());
		}
	}
	
	template<typename Traits>
	bool DistanceOperation<Traits>::setDataItem(DataItem* item)
	{
		if(!item)
			return false;
		
		_data_for_distance = dynamic_cast<DataForDistance<Traits>*>(item);
		
		if(!_data_for_distance)
			return false;
		
		_data_for_distance->incrementUseCount();
		return true;
	}
	
	template<typename Traits>
			OperationInterface* DistanceOperation<Traits>::clone()
	{
		return new DistanceOperation<Traits>();
	}
	
	/*
	 * This method handles the template parameters profdist::rna_structure_traits
	 * and profdist::protein_traits. In this cases it doesn't provide the kimura-2-parameter
	 * correction model. In the implementation file a specialized method is
	 * provided for the profdist::rna_traits case where we can provide a
	 * kimura-2-parameter correction model.
	 */
	template<typename Traits>
	bool DistanceOperation<Traits>::preExecution()
	{
		DistanceDialog dialog(mainWindow, true);
		int w = dialog.labelAlignmentIcon->width();
		int h = dialog.labelAlignmentIcon->height();
		dialog.labelAlignmentIcon->setPixmap(_data_for_distance->icon().pixmap(w, h));
		dialog.labelAlignmentName->setText(_data_for_distance->name());
		
		if(QDialog::Rejected == dialog.exec())
		{
			return false;
		}
		
		_correction = dialog.correctionModel();
		_matrix_file_name = dialog.lineEditRateMatrix->text();
		
		return true;
	}
	
#ifdef Q_OS_WIN32
	template<>
	bool DistanceOperation<profdist::rna_traits>::preExecution();
#endif
	
	template<typename Traits>
	bool DistanceOperation<Traits>::loadRateMatrix(const QString& fileName, typename Traits::rate_matrix& matrix)
	{
		if(fileName.isNull())
			return false;
		
		ifstream infile(fileName.toStdString().c_str());
		if(!infile)
		{
			GUI_MSG_N("could not open file to read rate matrix");
			return false;
		}
		if(!profdist::read_rate_matrix<Traits>(infile, matrix) && !infile.fail())
		{
			GUI_MSG_N("error while loading rate matrix");
			return false;
		}
		return true;
	}
	
}

#endif // PROFDIST_QTGUI_DISTANCE_OPERATION_HPP_
